<?php

use FirstpointCh\Shop\Models\Attribute;
use FirstpointCh\Shop\Models\Product;
use FirstpointCh\Shop\Models\ProductType;
use FirstpointCh\Shop\Models\Variant;
use Inertia\Testing\AssertableInertia;

it('shows the variant edit form', function () {
    $product = Product::factory()->create();
    $variant = Variant::factory()->create([
        'product_id' => $product->id,
    ]);

    $this->get('/cp/products/'.$product->id.'/variants/'.$variant->id.'/edit')
        ->assertStatus(200)
        ->assertInertia(fn (AssertableInertia $page) => $page
            ->component('ProductVariant/Form')
            ->where('product.id', $product->id)
            ->where('variant.id', $variant->id)
        );
});

it('has default attributes based on product type', function () {
    $productType = ProductType::factory()->create();
    $attribute = Attribute::factory()->create();

    Attribute::factory()->create(); // This attribute should not be added by default

    $productType->variantAttributes()->attach($attribute, ['type' => 'variant']);

    $product = Product::factory()->create([
        'product_type_id' => $productType->id,
    ]);

    $variant = Variant::factory()->create([
        'product_id' => $product->id,
    ]);

    $this->get('/cp/products/'.$product->id.'/variants/'.$variant->id.'/edit')
        ->assertInertia(fn (AssertableInertia $page) => $page
            ->component('ProductVariant/Form')
            ->has('variant.base_attributes', 1)
            ->whereAll([
                'variant.base_attributes.0.attribute_id' => $attribute->id,
                'variant.base_attributes.0.name' => $attribute->name,
            ])
        );
});

// TODO: Test that the default attribute value is propertly set according to the attribute type

it('has required fields', function () {
    $product = Product::factory()->create();
    $variant = Variant::factory()->create([
        'product_id' => $product->id,
    ]);

    $this->post('/cp/products/'.$product->id.'/variants/'.$variant->id, [
        'base_attributes' => [[]],
        'extra_attributes' => [[]],
        'prices' => [[]],
    ])->assertSessionHasErrors([
        'name',
        'slug',
        'track_stock',
        'base_attributes.0.attribute_id',
        'extra_attributes.0.attribute_id',
        'prices.0.region',
    ]);
});

test('track_stock must be boolean', function () {
    $product = Product::factory()->create();
    $variant = Variant::factory()->create([
        'product_id' => $product->id,
    ]);

    $this->post('/cp/products/'.$product->id.'/variants/'.$variant->id, ['track_stock' => 'test'])
        ->assertSessionHasErrors(['track_stock']);

    $this->post('/cp/products/'.$product->id.'/variants/'.$variant->id, ['track_stock' => true])
        ->assertSessionDoesntHaveErrors(['track_stock']);
});

test('each attribute must exist', function () {
    $product = Product::factory()->create();
    $variant = Variant::factory()->create([
        'product_id' => $product->id,
    ]);
    $attribute = Attribute::factory()->create();

    $this->post('/cp/products/'.$product->id.'/variants/'.$variant->id, [
        'base_attributes' => [
            ['attribute_id' => $attribute->id],
            ['attribute_id' => 5],
        ],
        'extra_attributes' => [
            ['attribute_id' => $attribute->id],
            ['attribute_id' => 5],
        ],
    ])
    ->assertSessionDoesntHaveErrors(['base_attributes.0.attribute_id', 'extra_attributes.0.attribute_id'])
    ->assertSessionHasErrors(['base_attributes.1.attribute_id', 'extra_attributes.1.attribute_id']);
});

test('price is required for every active regions', function () {
    $product = Product::factory()->create();
    $variant = Variant::factory()->create([
        'product_id' => $product->id,
    ]);

    $this->post('/cp/products/'.$product->id.'/variants/'.$variant->id, [
        'prices' => [
            ['slug' => 'ch', 'is_active' => 1, 'price' => null],
            ['slug' => 'fr', 'is_active' => 0, 'price' => null],
        ],
    ])
    ->assertSessionHasErrors(['prices.0.price'])
    ->assertSessionDoesntHaveErrors(['prices.1.price']);
});

it('updates a product variant', function () {
    $product = Product::factory()->create();
    $variant = Variant::factory()->create([
        'product_id' => $product->id,
    ]);
    $baseAttribute = Attribute::factory()->create(['type' => 'options']);
    $baseAttributeValue = $baseAttribute->options()->create(['value' => 'test']);
    $extraAttribute = Attribute::factory()->create(['type' => 'options']);
    $extraAttributeValue = $extraAttribute->options()->create(['value' => 'test extra']);

    $this->post('/cp/products/'.$product->id.'/variants/'.$variant->id, [
        'is_active' => false,
        'name' => ['en' => 'Test'],
        'reference' => 'test',
        'slug' => ['en' => 'test'],
        'track_stock' => false,
        'stock' => null,
        'base_attributes' => [
            ['attribute_id' => $baseAttribute->id, 'value' => $baseAttributeValue->id],
        ],
        'extra_attributes' => [
            ['attribute_id' => $extraAttribute->id, 'value' => $extraAttributeValue->id],
        ],
        'prices' => [
            ['region' => 'ch', 'is_active' => 1, 'price' => 129],
        ],
    ])
    ->assertStatus(302)
    ->assertRedirect('/cp/products/'.$product->id.'/variants');

    $this->assertDatabaseHas('variants', [
        'product_id' => $product->id,
        'is_active' => false,
        'name' => json_encode(['en' => 'Test']),
        'reference' => 'test',
        'track_stock' => false,
        'stock' => null,
        'slug' => json_encode(['en' => 'test']),
    ]);

    $variant = Variant::first();

    $variant->load('attributes');

    $this->assertCount(2, $variant->getRelation('attributes'));
    $this->assertEquals($baseAttributeValue->id, $variant->getRelation('attributes')->first()->pivot->value->raw());
    $this->assertEquals($extraAttributeValue->id, $variant->getRelation('attributes')->last()->pivot->value->raw());
    $this->assertCount(1, $variant->prices);
    $this->assertEquals(129, $variant->prices->first()->price);
});
