<?php

namespace FirstpointCh\Shop;

use FirstpointCh\Shop\FieldTypes\Boolean;
use FirstpointCh\Shop\FieldTypes\File;
use FirstpointCh\Shop\FieldTypes\Number;
use FirstpointCh\Shop\FieldTypes\Text;
use FirstpointCh\Shop\FieldTypes\Tiptap;
use FirstpointCh\Shop\Region;
use Illuminate\Support\Collection;

class Shop
{
    protected ?Region $currentRegion = null;

    protected $regionAware = false;

    private $cpScripts = [];

    private $attributeTypes = [];

    private $fieldTypes = [
        Text::class,
        Number::class,
        Boolean::class,
        Tiptap::class,
        File::class,
    ];

    public function __construct(string $region)
    {
        $this->setRegion($region);
    }

    public function getRegion()
    {
        return $this->currentRegion;
    }

    public function setRegion($region)
    {
        if (!config()->has("shop.regions.{$region}")) {
            throw new \Exception('Invalid region "'.$region.'"');
        }

        if ($region instanceof Region) {
            $this->currentRegion = $region;
        } elseif (is_string($region)) {
            if (!empty($this->currentRegion) && $region === $this->currentRegion->slug) {
                return;
            }

            $regionData = config("shop.regions.{$region}");

            $this->currentRegion = new Region(
                name: $regionData['name'],
                slug: $regionData['slug'],
                currency: $regionData['currency'],
                currency_format: $regionData['currency_format'],
                countries: $regionData['countries'],
            );
        } else {
            throw new \Exception('Invalid region "'.$region.'"');
        }
    }

    public function isRegionAware()
    {
        return $this->regionAware;
    }

    public function regionAware($value = true)
    {
        $this->regionAware = $value;

        return $this;
    }

    public function registerScripts(array $paths)
    {
        $this->cpScripts = array_merge($this->cpScripts, $paths);
    }

    public function getCpScripts()
    {
        return $this->cpScripts;
    }

    public function registerAttributeType($attributeType)
    {
        $this->attributeTypes[$attributeType::$fieldType] = $attributeType;
    }

    public function getAttributeTypes()
    {
        return collect($this->attributeTypes)
            ->map(fn ($type, $key) => ['name' => $key, 'label' => $type->label()]);
    }

    public function getFieldTypes(): Collection
    {
        return collect($this->fieldTypes)->map(fn ($type) => (new $type)->toArray());
    }

    public function registerFieldType($class)
    {
        $this->fieldTypes[] = $class;
    }
}
