<?php

namespace FirstpointCh\Shop\Models;

use FirstpointCh\Shop\Cart\Orderable;
use FirstpointCh\Shop\Catalog\QueryBuilders\VariantQueryBuilder;
use FirstpointCh\Shop\Models\CatalogIndex as Index;
use FirstpointCh\Shop\Traits\HasCustomFields;
use FirstpointCh\Shop\Traits\HasPackageFactory;
use FirstpointCh\Translatable\Casts\Localized;
use FirstpointCh\Translatable\Traits\Translatable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\DB;

/**
 * @method static \FirstpointCh\Shop\Catalog\QueryBuilders\VariantQueryBuilder query()
 */
class Variant extends Model implements Orderable
{
    use HasPackageFactory, Translatable, SoftDeletes, HasCustomFields;

    protected $guarded = [];

    protected $casts = [
        'is_active' => 'boolean',
        'name' => Localized::class,
        'slug' => Localized::class,
        'track_stock' => 'boolean',
        'is_shippable' => 'boolean',
        'meta_title' => Localized::class,
        'meta_description' => Localized::class,
    ];

    protected static function booted()
    {
        static::addGlobalScope('region-aware', function ($builder) {
            $builder->when(shop()->isRegionAware(), function ($query) {
                $query->whereRelation('prices', 'region', shop()->getRegion()->slug);
            });
        });
    }

    public function newEloquentBuilder($query)
    {
        return new VariantQueryBuilder($query);
    }

    public function resolveRouteBinding($value, $field = null)
    {
        if ($field === 'slug') {
            return $this->where('slug->'.app()->getLocale(), $value)->first()
                ?? $this->where('slug->'.config('app.fallback_locale'), $value)->firstOrFail();
        }

        return $this->resolveRouteBindingQuery($this, $value, $field)->first();
    }

    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    public function attributes(): BelongsToMany
    {
        return $this->belongsToMany(Attribute::class)
            ->withPivot(['type', 'value', 'translatable'])
            ->withTimestamps()
            ->using(AttributeVariant::class);
    }

    public function prices(): HasMany
    {
        return $this->hasMany(VariantPrice::class);
    }

    public function loadPrice($region = null)
    {
        $price = $this->prices()
            ->where('region', $region ?? shop()->getRegion()->slug)
            ->first();

        if (! $price) {
            return null;
        }

        $this->setAttribute('price', $price->pivot->price);

        return $price;
    }

//    public function routes()
//    {
//        return $this->morphMany(Route::class, 'routable');
//    }

    public function isOrderable($quantity = 1): bool
    {
        return $this->is_active
            && ! $this->isOutOfStock($quantity);
    }

    public function isOutOfStock($quantity = 1): bool
    {
        if (! $this->track_stock) {
            return false;
        }

        return $this->stock - $quantity < 0;
    }

    public function index(Product $product = null)
    {
        if ($this->trashed()) {
            return;
        }

        $product = $product ?? $this->product;

        $product->loadMissing('attributes');
        $this->loadMissing('attributes');

        DB::transaction(function () use ($product) {
            foreach ($this->prices as $price) {
                Index::withoutGlobalScope('region-aware')->updateOrCreate([
                    'id' => $price->region.'-'.$product->id.'-'.$this->id,
                ], [
                    'product_id' => $product->id,
                    'region' => $price->region,
                    'variant_id' => $this->id,
                    'product_name' => $product->name,
                    'product_status' => $product->status,
                    'product_categories' => $product->categories->pluck('slug'),
                    'product_brand_id' => $product->brand_id,
                    'variant_sku' => $this->sku,
                    'variant_reference' => $this->reference,
                    'variant_name' => $this->name,
                    'variant_is_active' => $this->is_active,
                    'price' => $price->price,
                    'product_attributes' => $product->getRelation('attributes')->mapWithKeys(fn ($attribute) => [
                        $attribute->key => $attribute->pivot->value->raw(),
                    ]),
                    'variant_attributes' => $this->getRelation('attributes')->mapWithKeys(fn ($attribute) => [
                        $attribute->key => $attribute->pivot->value->raw(),
                    ]),
                ]);
            }
        });
    }
}
