<?php

namespace FirstpointCh\Shop\Models;

use DB;
use FirstpointCh\Shop\Traits\HasPackageFactory;
use FirstpointCh\Translatable\Casts\Localized;
use FirstpointCh\Translatable\Traits\Translatable;
use Illuminate\Database\Eloquent\Model;

class ShippingMethod extends Model
{
    use HasPackageFactory, Translatable;

    protected $guarded = [];

    protected $casts = [
        'name' => Localized::class,
        'description' => Localized::class,
    ];

    protected static function booted()
    {
        static::addGlobalScope('region-aware', function ($builder) {
            $builder->when(shop()->isRegionAware(), function ($query) {
                $query->whereRelation('prices', 'region', shop()->getRegion()->slug);
            });
        });
    }

    public function prices()
    {
        return $this->hasMany(ShippingMethodPrice::class);
    }

    public function scopeRestrictByWeight($query, $weight)
    {
        return $query->where(function ($query) use ($weight) {
            $query->whereNull('min_weight')
                ->orWhere('min_weight', '<=', $weight);
        })->where(function ($query) use ($weight) {
            $query->whereNull('max_weight')
                ->orWhere('max_weight', '>=', $weight);
        });
    }

    public function scopeRestrictByCartPrice($query, $price)
    {
        return $query
            ->whereHas('prices', function ($query) use ($price) {
                $query->where(function ($query) use ($price) {
                    $query->whereNull('min_price')->orWhere('min_price', '<=', $price);
                })->where(function ($query) use ($price) {
                    $query->whereNull('max_price')->orWhere('max_price', '>=', $price);
                });
            });
    }

    public function scopeWithPrice($query, $cartPrice = null)
    {
        return $query->addSelect(['price' => DB::table('shipping_method_prices')
            ->select('price')
            ->whereColumn('shipping_method_id', 'shipping_methods.id')
            ->when($cartPrice, function ($query) use ($cartPrice) {
                $query->where(function ($query) use ($cartPrice) {
                    $query->whereNull('min_price')->orWhere('min_price', '<=', $cartPrice);
                })->where(function ($query) use ($cartPrice) {
                    $query->whereNull('max_price')->orWhere('max_price', '>=', $cartPrice);
                });
            })
            ->limit(1),
        ]);
    }
}
