<?php

namespace FirstpointCh\Shop\Http\Controllers\Cp;

use FirstpointCh\Shop\Http\Controllers\Controller;
use FirstpointCh\Shop\Http\Requests\DatatableRequest;
use FirstpointCh\Shop\Models\ProductType;
use FirstpointCh\Shop\Region;
use FirstpointCh\Shop\Models\TaxRule;
use Illuminate\Support\Arr;
use Inertia\Inertia;

class TaxRuleController extends Controller
{
    public function index(DatatableRequest $datatable)
    {
        return Inertia::render('TaxRule/Index', [
            'taxRules' => $datatable
                ->searchable(['name', 'description'])
                ->query(TaxRule::with('regions'))
                ->get(),
            'datatableState' => $datatable->getState(),
        ]);
    }

    public function create()
    {
        return Inertia::render('TaxRule/Form', [
            'regions' => Region::all()->only('slug', 'name'),
            'productTypes' => ProductType::select('id', 'name')->get(),
        ]);
    }

    public function store()
    {
        TaxRule::create(request()->validate([
            'name' => 'required|array',
            'description' => 'required|array',
            'rate' => 'required|numeric',
            'tax_included' => 'boolean',
        ]));

        return redirect('/cp/tax-rules');
    }

    public function edit(TaxRule $taxRule)
    {
        $taxRule->in('*');

        $taxRule->setRelation('regions', $taxRule->regions()->pluck('region'));
        $taxRule->setRelation('product_types', $taxRule->productTypes()->pluck('product_types.id'));

        return Inertia::render('TaxRule/Form', [
            'taxRule' => $taxRule,
            'regions' => Region::all(),
            'productTypes' => ProductType::select('id', 'name')->get(),
        ]);
    }

    public function update(TaxRule $taxRule)
    {
        $data = request()->validate([
            'name' => 'required|array',
            'description' => 'required|array',
            'rate' => 'required|numeric',
            'tax_included' => 'boolean',
            'apply_to_all_regions' => 'boolean',
            'regions' => 'nullable|array',
            'apply_to_all_product_types' => 'boolean',
            'product_types' => 'nullable|array',
        ]);

        $regions = Arr::pull($data, 'regions');
        $productTypes = Arr::pull($data, 'product_types');

        $taxRule->update($data);

        if ($taxRule->apply_to_all_regions) {
            $taxRule->regions()->delete();
        } else {
            $taxRule->regions()->whereNotIn('region', $regions)->delete();

            foreach ($regions as $region) {
                $taxRule->regions()->updateOrCreate(['region' => $region]);
            }
        }

        $taxRule->apply_to_all_product_types
            ? $taxRule->productTypes()->detach()
            : $taxRule->productTypes()->sync($productTypes);

        return redirect('/cp/tax-rules');
    }

    public function destroy(TaxRule $taxRule)
    {
        $taxRule->delete();

        return redirect('/cp/tax-rules');
    }
}
