<?php

namespace FirstpointCh\Shop\Http\Controllers\Cp;

use FirstpointCh\Shop\Http\Controllers\Controller;
use FirstpointCh\Shop\Http\Requests\DatatableRequest;
use FirstpointCh\Shop\Models\Customer;
use FirstpointCh\Shop\Region;
use Inertia\Inertia;

class CustomerController extends Controller
{
    public function index(DatatableRequest $datatable)
    {
        return Inertia::render('Customer/Index', [
            'customers' => $datatable
                ->searchable(['first_name', 'last_name', 'email'])
                ->query(
                    Customer::query()
                        ->when(
                            request('filters.region'),
                            fn ($query, $region) => $query->whereHas('orders', function ($query) use ($region) {
                                $query->where('region', $region);
                            })
                        )
                )
                ->get(),
            'datatableState' => $datatable->getState(),
            'regions' => Region::all(),
        ]);
    }

    public function create()
    {
        return Inertia::render('Customer/Form');
    }

    public function store()
    {
        Customer::create(request()->validate([
            'first_name' => 'required|max:255',
            'last_name' => 'required|max:255',
            'email' => 'required|max:255|email|unique:customers,email',
            'password' => 'required',
        ]));

        return redirect('/cp/customers');
    }

    public function show(Customer $customer, string $tab = null)
    {
        return Inertia::render('Customer/Show', [
            'tab' => $tab,
            'customer' => $customer,
            'addresses' => (new DatatableRequest)->query($customer->addresses())->get(),
            'carts' => (new DatatableRequest)->query($customer->carts())->get(),
            'orders' => (new DatatableRequest)->query($customer->orders())->get(),
            'emails' => (new DatatableRequest)->query($customer->notificationLogs())->get(),
        ]);
    }

    public function edit(Customer $customer)
    {
        return Inertia::render('Customer/Form', [
            'customer' => $customer,
        ]);
    }

    public function update(Customer $customer)
    {
        $data = request()->validate([
            'first_name' => 'required|max:255',
            'last_name' => 'required|max:255',
            'email' => 'required|max:255|email|unique:customers,email,'.$customer->id,
            'password' => 'nullable',
        ]);

        if (empty($data['password'])) {
            unset($data['password']);
        } else {
            $data['password'] = bcrypt($data['password']);
        }

        $customer->update($data);

        return redirect('/cp/customers');
    }
}
