<?php

namespace FirstpointCh\Shop\Data\Cp;

use FirstpointCh\Shop\Models\Attribute;
use Spatie\LaravelData\Data;

class ProductVariantFormData extends Data
{
    public function __construct(
        public ?bool $is_active,
        public array $name,
        public array $slug,
        public string $reference,
        public bool $track_stock,
        public ?string $sku,
        public ?int $stock,
        public bool $is_shippable = true,
        public ?float $weight_grams = null,
        public ?array $base_attributes = [],
        public ?array $extra_attributes = [],
        public ?array $prices = [],
        public ?array $custom_fields = [],
        public ?array $meta_title = [],
        public ?array $meta_description = [],
    ) {
    }

    public static function rules(): array
    {
        return [
            'is_active' => 'nullable',
            'name' => 'required',
            'slug' => 'required',
            'reference' => 'required',
            'track_stock' => 'required|boolean',
            'sku' => 'nullable',
            'stock' => 'nullable|numeric',
            'is_shippable' => 'boolean',
            'weight_grams' => 'nullable|numeric',
            'base_attributes' => 'array',
            'base_attributes.*.attribute_id' => 'required|exists:attributes,id',
            'base_attributes.*.value' => 'nullable',
            'extra_attributes' => 'array',
            'extra_attributes.*.attribute_id' => 'required|exists:attributes,id',
            'extra_attributes.*.value' => 'nullable',
            'prices' => 'array',
            'prices.*.region' => 'required',
            'prices.*.is_active' => 'nullable|boolean',
            'prices.*.price' => 'required_if:prices.*.is_active,1',
            'custom_fields' => 'nullable|array',
            'meta_title' => 'nullable|array',
            'meta_description' => 'nullable|array',
        ];
    }

    public function getVariantData()
    {
        return [
            'is_active' => $this->is_active,
            'name' => $this->name,
            'slug' => $this->slug,
            'reference' => $this->reference,
            'track_stock' => $this->track_stock,
            'sku' => $this->sku,
            'stock' => $this->stock,
            'is_shippable' => $this->is_shippable,
            'weight_grams' => $this->weight_grams,
            'meta_title' => $this->meta_title,
            'meta_description' => $this->meta_description,
        ];
    }

    public function getAttributes()
    {
        $baseAttributes = collect($this->base_attributes)
            ->mapWithKeys(function ($attr) {
                $attribute = Attribute::find($attr['attribute_id']);

                return [
                    $attr['attribute_id'] => [
                        'value' => $attr['value'],
                        'type' => $attribute->type,
                        'translatable' => $attribute->translatable,
                    ],
                ];
            })->toArray();

        $extraAttributes = collect($this->extra_attributes)
            ->mapWithKeys(function ($attr) {
                $attribute = Attribute::find($attr['attribute_id']);

                return [
                    $attr['attribute_id'] => [
                        'value' => $attr['value'],
                        'type' => $attribute->type,
                        'translatable' => $attribute->translatable,
                    ],
                ];
            })->toArray();

        return $baseAttributes + $extraAttributes; // Merge but keep keys
    }

    public function getPrices($productId)
    {
        return collect($this->prices)
            ->where('is_active', true)
            ->map(fn ($price) => [
                'region' => $price['region'],
                'product_id' => $productId,
                'price' => $price['price'],
            ]);
    }
}
