<?php

namespace FirstpointCh\Shop\Catalog\QueryBuilders;

use FirstpointCh\Shop\Models\Attribute;
use FirstpointCh\Shop\Models\VariantPrice;
use Illuminate\Database\Eloquent\Builder;

class VariantQueryBuilder extends Builder
{
    public function withPrice(): self
    {
        return $this->addSelect(['price' => VariantPrice::select('price')
            ->where('region', shop()->getRegion()->slug)
            ->whereColumn('variant_id', 'variants.id')
            ->limit(1),
        ]);
    }

    public function active(bool $active = true): self
    {
        return $this->where('is_active', $active);
    }

    public function whereAttributeIn(string $type, string $attribute, array $values, ?string $locale = null): self
    {
        return $this->whereHas('attributes', function ($query) use ($type, $attribute, $values, $locale) {
            $query->where('attributes.key', $attribute);

            foreach ($values as $value) {
                $query->orWhere('attribute_variant.value', $value);
            }
        });
    }

    public function wherePriceBetween(?float $min, ?float $max): self
    {
        return $this->whereHas('regions', function ($query) use ($min, $max) {
            $query
                ->when(shop()->isRegionAware(), fn ($query) => $query->where('region', shop()->getRegion()->slug))
                ->when($min, fn ($query) => $query->where('price', '>=', $min))
                ->when($max, fn ($query) => $query->where('price', '<=', $max));
        });
    }

    public function whereAttribute(Attribute $attribute, $value): self
    {
        return $this->whereHas('attributes', function ($query) use ($attribute, $value) {
            $query->where('attributes.key', $attribute->key)
                ->where('attribute_variant.value', $value);
        });
    }
}
