<?php

namespace FirstpointCh\Shop\Catalog\QueryBuilders;

use FirstpointCh\Shop\Models\Attribute;
use Illuminate\Database\Eloquent\Builder;

class ProductQueryBuilder extends Builder
{
    public function published(): self
    {
        return $this->where('status', 'published');
    }

    public function whereCategoriesIn(?array $categories = []): self
    {
        if (empty($categories)) {
            return $this;
        }

        return $this->whereHas('categories', fn ($query) => $query->whereIn('categories.id', $categories));
    }

    public function wherePriceBetween(?float $min = null, ?float $max = null): self
    {
        return $this->whereHas('regions', function ($query) use ($min, $max) {
            $query
                ->when(shop()->isRegionAware(), fn ($query) => $query->where('region', shop()->getRegion()->slug))
                ->when($min, fn ($query) => $query->where('price', '>=', $min))
                ->when($max, fn ($query) => $query->where('price', '<=', $max));
        });
    }

    public function whereAttributeIn(string $type, string $attribute, array $values, ?string $locale = null): self
    {
        return $this->whereHas('attributes', function ($query) use ($type, $attribute, $values, $locale) {
            $query->where('attributes.key', $attribute);

            foreach ($values as $value) {
                $query->orWhere('attribute_product.value', $value);
            }
        });
    }

    public function whereAttribute(Attribute $attribute, $value): self
    {
        return $this->whereHas('attributes', function ($query) use ($attribute, $value) {
            $query->where('attributes.key', $attribute->key)
                ->where('attribute_product.value', $value);
        });
    }

    public function withTotalTaxRate($type = null): self
    {
        return $this->addSelect(['total_tax_rate' => function ($query) use ($type) {
            $query->selectRaw('SUM(tax_rules.rate)')
                ->from('tax_rules')
                ->when($type === 'included', function ($query) {
                    $query->where('tax_rules.tax_included', true);
                })
                ->when($type === 'excluded', function ($query) {
                    $query->where('tax_rules.tax_included', false);
                })
                ->where(function ($query) {
                    // The tax rule must be applied to all regions or the current region
                    $query->where('tax_rules.apply_to_all_regions', true)
                        ->orWhereExists(function ($query) {
                            $query->selectRaw(1)
                                ->from('tax_rule_regions')
                                ->whereColumn('tax_rule_regions.tax_rule_id', 'tax_rules.id')
                                ->where('tax_rule_regions.region', shop()->getRegion()->slug);
                        });
                })
                ->where(function ($query) {
                    // The tax rule must be applied to all product types or this product type
                    $query->where('tax_rules.apply_to_all_product_types', true)
                        ->orWhereExists(function ($query) {
                            $query->selectRaw(1)
                                ->from('product_type_tax_rule')
                                ->whereColumn('product_type_tax_rule.tax_rule_id', 'tax_rules.id')
                                ->whereColumn('product_type_tax_rule.product_type_id', 'products.product_type_id');
                        });
                });
        }])->withCasts([
            'total_tax_rate' => 'float',
        ]);
    }
}
