<?php

namespace FirstpointCh\Shop\Models;

use App\Casts\Json;
use FirstpointCh\Shop\Catalog\QueryBuilders\ProductQueryBuilder;
use FirstpointCh\Shop\Traits\HasCustomFields;
use FirstpointCh\Shop\Traits\HasPackageFactory;
use FirstpointCh\Translatable\Casts\Localized;
use FirstpointCh\Translatable\Traits\Translatable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Spatie\MediaLibrary\MediaCollections\Models\Media;

/**
 * @method static \FirstpointCh\Shop\Catalog\QueryBuilders\ProductQueryBuilder query()
 */
class Product extends Model implements HasMedia
{
    use HasPackageFactory, Translatable, InteractsWithMedia, HasCustomFields;

    protected $guarded = [];

    protected $casts = [
        'is_active' => 'boolean',
        'name' => Localized::class,
        'short_description' => Localized::class,
        'description' => Localized::class,
    ];

    protected static function booted()
    {
        static::addGlobalScope('region-aware', function ($builder) {
            $builder->when(shop()->isRegionAware(), function ($query) {
                $query->whereRelation('regions', 'region_id', shop()->getRegion()->id);
            });
        });
    }

    public function newEloquentBuilder($query): ProductQueryBuilder
    {
        return new ProductQueryBuilder($query);
    }

    public function registerMediaCollections(): void
    {
        $this->addMediaCollection('pictures')
            ->useDisk(config('shop.media.product.disk'))
            ->registerMediaConversions(function (Media $media) {
                $this->addMediaConversion('cp_thumbnail')->crop('crop-center', 150, 150);
                $this->addMediaConversion('cp_tiny')->crop('crop-center', 50, 50);

                foreach (config('shop.media.product.conversions') as $name => $conversion) {
                    $this->addMediaConversion($name)
                        ->crop('crop-center', $conversion['width'], $conversion['height'])
                        ->withResponsiveImages();
                }
            });

//        $this->addMediaCollection('documents');
    }

    public function categories(): BelongsToMany
    {
        return $this->belongsToMany(Category::class);
    }

    public function brand(): BelongsTo
    {
        return $this->belongsTo(Brand::class);
    }

    public function taxRules(): BelongsToMany
    {
        return $this->belongsToMany(TaxRule::class)
            ->withTimestamps();
    }

    public function regions(): BelongsToMany
    {
        return $this->belongsToMany(Region::class, 'region_variant', 'product_id', 'region_id');
    }

    public function variants(): HasMany
    {
        return $this->hasMany(Variant::class)
            ->withPrice();
    }

    public function attributes(): BelongsToMany
    {
        return $this->belongsToMany(Attribute::class)
            ->withPivot(['type', 'value', 'translatable'])
            ->withTimestamps()
            ->using(AttributeProduct::class);
    }

    public function productType(): BelongsTo
    {
        return $this->belongsTo(ProductType::class);
    }

    public function index()
    {
        \DB::transaction(function () {
            $this->variants()->withoutGlobalScope('region-aware')->get()->map->index($this);
        });
    }
}
