<?php

namespace FirstpointCh\Shop\Http\Controllers\Cp;

use FirstpointCh\Shop\Data\Cp\DefaultProductAttributeData;
use FirstpointCh\Shop\Data\Cp\ProductAttributeData;
use FirstpointCh\Shop\Data\Cp\ProductFormData;
use FirstpointCh\Shop\Http\Controllers\Controller;
use FirstpointCh\Shop\Http\Requests\DatatableRequest;
use FirstpointCh\Shop\Models\Attribute;
use FirstpointCh\Shop\Models\Brand;
use FirstpointCh\Shop\Models\Category;
use FirstpointCh\Shop\Models\CustomField;
use FirstpointCh\Shop\Models\Product;
use FirstpointCh\Shop\Models\ProductType;
use FirstpointCh\Shop\Models\TaxRule;
use FirstpointCh\Shop\Models\Variant;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;
use Spatie\MediaLibrary\MediaCollections\Models\Media;

class ProductController extends Controller
{
    public function index(DatatableRequest $datatable)
    {
        return Inertia::render('Product/Index', [
            'products' => $datatable
                ->searchable(['name'])
                ->query(Product::with('productType'))
                ->transform(fn ($results) => $results
                    ->map(fn ($item) => $item
                        ->setAttribute('thumbnail', $item->getFirstMediaUrl('pictures', 'cp_tiny'))))
                ->get(),
            'datatableState' => $datatable->getState(),
            'productTypes' => ProductType::all(),
        ]);
    }

    public function create()
    {
        $product = new Product([
            'status' => 'draft',
            'name' => [config('app.fallback_locale') => null],
            'product_type_id' => request('productTypeId'),
            'short_description' => [config('app.fallback_locale') => null],
            'description' => [config('app.fallback_locale') => null],
            'base_attributes' => [],
            'extra_attributes' => [],
            'brand_id' => null,
            'custom_fields' => [],
        ]);

        if (! empty($product->productType)) {
            $product->setAttribute('base_attributes', DefaultProductAttributeData::collection(
                $product->productType->productAttributes
            ));
        }

        $product->setRelation('categories', collect());

        return Inertia::render('Product/Form', [
            'product' => $product->in('*'),
            'productTypes' => ProductType::all(),
            'categories' => Category::whereNull('parent_id')->with('children')->get(),
            'brands' => Brand::all(),
            'customFields' => CustomField::all(),
        ]);
    }

    public function store(ProductFormData $data)
    {
        $product = DB::transaction(function () use ($data) {
            $product = Product::create($data->getProductData());
            $product->attributes()->sync($data->getAttributes());
            $product->categories()->sync($data->categories);
            $product->updateCustomFields($data->custom_fields);
            $product->index();

            return $product;
        });

        return redirect('/cp/products/'.$product->id)->with('toast', [
            'title' => 'Produit créé',
            'message' => 'Le produit a bien été créé',
            'type' => 'success',
        ]);
    }

    public function show(DatatableRequest $datatable, Product $product, string $tab = null)
    {
        $product->load(['attributes']);

        $product->setRelation('categories', $product->categories->pluck('name'));

        return Inertia::render('Product/Show', [
            'tab' => $tab,
            'product' => $product,
            'variants' => $datatable
                ->searchable(['name'])
                ->query(
                    Variant::query()
                        ->where('product_id', $product->id)
                        ->when(request('filters.trashed') === 'yes', fn ($query) => $query->withTrashed())
                        ->when(request('filters.trashed') === 'only', fn ($query) => $query->onlyTrashed())
                )
                ->get(),
            'datatableState' => $datatable->getState(),
            'taxRules' => TaxRule::forProduct($product)->paginate(10),
            'pictures' => $product->getMedia('pictures')->map(fn (Media $media) => [
                'id' => $media->id,
                'name' => $media->name,
                'original_url' => $media->getFullUrl(),
                'thumbnail_url' => $media->getFullUrl('cp_thumbnail'),
                'alt_text' => $media->getCustomProperty('alt_text'),
                'size' => $media->size,
            ]),
        ]);
    }

    public function edit(Product $product)
    {
        $baseAttributeIds = $product->productType
            ? $product->productType->productAttributes->pluck('id')->toArray()
            : [];

        if (! empty($product->productType)) {
            $baseAttributes = $product->attributes()->whereIn('attributes.id', $baseAttributeIds)->get();

            $product->setAttribute('base_attributes', $product->productType->productAttributes->map(fn ($attribute) => [
                'attribute_id' => $attribute->id,
                'name' => $attribute->name,
                'type' => $attribute->type,
                'translatable' => $attribute->translatable,
                'options' => $attribute->options->map->in('*'),
                'value' => $attribute->translatable
                    ? $baseAttributes->firstWhere('id', $attribute->id)?->pivot->value->raw() ?? [config('app.fallback_locale') => null]
                    : $baseAttributes->firstWhere('id', $attribute->id)?->pivot->value->raw(),
            ]));
        }

        $product->setAttribute('extra_attributes', ProductAttributeData::collection(
            $product->attributes()->whereNotIn('attributes.id', $baseAttributeIds)->get()->map(fn ($attribute) => [
                'attribute_id' => $attribute->id,
                'name' => $attribute->name,
                'type' => $attribute->type,
                'translatable' => $attribute->translatable,
                'options' => $attribute->options->map->in('*'),
                'value' => $baseAttributes->firstWhere('id', $attribute->id)?->pivot->value->raw(),
            ])
        ));

        $categories = Category::whereNull('parent_id')->with('children')->get();

        $product->setRelation('categories', $product->categories->pluck('id'));
        $product->load(['customFields']);

        return Inertia::render('Product/Form', [
            'product' => $product->in('*'),
            'productTypes' => ProductType::all(),
            'productTypeAttributes' => $product->productType
                ? $product->productType->productAttributes->map->only(['id', 'name', 'type'])
                : [],
            'attributes' => Attribute::all(),
            'categories' => $categories,
            'brands' => Brand::all(),
            'customFields' => CustomField::all(),
        ]);
    }

    public function update(Product $product, ProductFormData $data)
    {
        DB::transaction(function () use ($product, $data) {
            $product->update($data->getProductData());
            $product->attributes()->sync($data->getAttributes());
            $product->categories()->sync($data->categories);
            $product->updateCustomFields($data->custom_fields);
            $product->index();
        });

        return redirect('/cp/products/'.$product->id)->with('toast', [
            'title' => 'Produit enregistré',
            'message' => 'Le produit a bien été enregistré',
            'type' => 'success',
        ]);
    }

    public function updateStatus(Product $product)
    {
        $data = request()->validate([
            'status' => 'required|in:draft,published',
        ]);

        $product = DB::transaction(function () use ($product, $data) {
            $product->update($data);
            $product->index();

            return $product;
        });

        return back()->with('toast', [
            'title' => 'Produit mis à jour',
            'message' => 'Le produit a bien été '.['draft' => 'désactivé', 'published' => 'publié'][$product->status],
            'type' => 'success',
        ]);
    }

    public function delete(Product $product)
    {
        $product->delete();

        return redirect('/cp/products')->with('toast', [
            'title' => 'Produit supprimé',
            'message' => 'Le produit a bien été supprimé',
            'type' => 'success',
        ]);
    }
}
