<?php

use FirstpointCh\Shop\Models\Channel;
use FirstpointCh\Shop\Models\ShippingMethod;
use Illuminate\Support\Arr;
use Inertia\Testing\AssertableInertia;

it('shows the shipping method edit form', function () {
    $shippingMethod = ShippingMethod::factory()->create();

    $this->get('/cp/shipping-methods/'.$shippingMethod->id.'/edit')
        ->assertStatus(200)
        ->assertInertia(fn (AssertableInertia $page) => $page
            ->component('ShippingMethod/Form')
            ->where('shippingMethod.id', $shippingMethod->id)
            ->has('channels', 1)
        );
});

test('the name is required and must be an array', function () {
    $id = ShippingMethod::factory()->create()->id;

    $this->post('/cp/shipping-methods/'.$id)->assertStatus(302)->assertSessionHasErrors('name');
    $this->post('/cp/shipping-methods/'.$id, ['name' => 'test'])->assertStatus(302)->assertSessionHasErrors('name');
    $this->post('/cp/shipping-methods/'.$id, ['name' => ['en' => 'test']])->assertSessionDoesntHaveErrors('name');
});

test('the description is required and must be an array', function () {
    $id = ShippingMethod::factory()->create()->id;

    $this->post('/cp/shipping-methods/'.$id)->assertStatus(302)->assertSessionHasErrors('description');
    $this->post('/cp/shipping-methods/'.$id, ['description' => 'test'])->assertStatus(302)->assertSessionHasErrors('description');
    $this->post('/cp/shipping-methods/'.$id, ['description' => ['en' => 'test']])->assertSessionDoesntHaveErrors('description');
});

test('the slug is required', function () {
    $id = ShippingMethod::factory()->create()->id;

    $this->post('/cp/shipping-methods/'.$id)->assertStatus(302)->assertSessionHasErrors('slug');
});

test('channels must be an array', function () {
    $id = ShippingMethod::factory()->create()->id;

    $this->post('/cp/shipping-methods/'.$id, ['channels' => 'test'])->assertStatus(302)->assertSessionHasErrors('channels');
    $this->post('/cp/shipping-methods/'.$id, ['channels' => []])->assertStatus(302)->assertSessionDoesntHaveErrors('channels');
});

test('each channel must exist and have a price', function () {
    $channel = Channel::factory()->create();
    $id = ShippingMethod::factory()->create()->id;

    $this->post('/cp/shipping-methods/'.$id, [
        'channels' => [
            ['id' => 123],
        ],
    ])->assertStatus(302)->assertSessionHasErrors('channels.0.id');

    $this->post('/cp/shipping-methods', [
        'channels' => [
            ['id' => $channel->id],
        ],
    ])->assertStatus(302)->assertSessionDoesntHaveErrors('channels.0.id');
});

it('updates a shipping method', function () {
    $shippingMethod = ShippingMethod::factory()->create();

    $data = [
        'name' => ['en' => 'My shipping method'],
        'description' => ['en' => 'My shipping method'],
        'slug' => 'my-shipping-method',
        'channels' => [
            ['id' => defaultChannel()->id, 'price' => 123],
        ],
    ];

    $this->assertDatabaseCount('channel_shipping_method', 0);

    $this->post('/cp/shipping-methods/'.$shippingMethod->id, $data)
        ->assertStatus(302)
        ->assertRedirect('/cp/shipping-methods');

    $channels = Arr::pull($data, 'channels');
    $data['name'] = json_encode($data['name']);
    $data['description'] = json_encode($data['description']);
    $data['id'] = $shippingMethod->id;

    $this->assertDatabaseCount('channel_shipping_method', 1);
    $this->assertDatabaseHas('shipping_methods', $data);
    $this->assertDatabaseHas('channel_shipping_method', [
        'channel_id' => $channels[0],
        'shipping_method_id' => ShippingMethod::first()->id,
        'price' => 123,
    ]);
});
