<?php

use FirstpointCh\Shop\Models\Channel;
use FirstpointCh\Shop\Models\ShippingMethod;
use Illuminate\Support\Arr;
use Inertia\Testing\AssertableInertia;

it('shows the shipping method create form', function () {
    $this->get('/cp/shipping-methods/create')
        ->assertStatus(200)
        ->assertInertia(fn (AssertableInertia $page) => $page
            ->component('ShippingMethod/Form')
            ->has('channels', 1)
        );
});

test('the name is required and must be an array', function () {
    $this->post('/cp/shipping-methods')->assertStatus(302)->assertSessionHasErrors('name');
    $this->post('/cp/shipping-methods', ['name' => 'test'])->assertStatus(302)->assertSessionHasErrors('name');
    $this->post('/cp/shipping-methods', ['name' => ['en' => 'test']])->assertSessionDoesntHaveErrors('name');
});

test('the description is required and must bne an array', function () {
    $this->post('/cp/shipping-methods')->assertStatus(302)->assertSessionHasErrors('description');
    $this->post('/cp/shipping-methods', ['description' => 'test'])->assertStatus(302)->assertSessionHasErrors('description');
    $this->post('/cp/shipping-methods', ['description' => ['en' => 'test']])->assertSessionDoesntHaveErrors('description');
});

test('the slug is required', function () {
    $this->post('/cp/shipping-methods')->assertStatus(302)->assertSessionHasErrors('slug');
});

test('channels must be an array', function () {
    $this->post('/cp/shipping-methods', ['channels' => 'test'])->assertStatus(302)->assertSessionHasErrors('channels');
    $this->post('/cp/shipping-methods', ['channels' => []])->assertStatus(302)->assertSessionDoesntHaveErrors('channels');
});

test('each channel must exist and have a price', function () {
    $channel = Channel::factory()->create();

    $this->post('/cp/shipping-methods', [
        'channels' => [
            ['id' => 123],
        ],
    ])->assertStatus(302)->assertSessionHasErrors('channels.0.id');

    $this->post('/cp/shipping-methods', [
        'channels' => [
            ['id' => $channel->id],
        ],
    ])->assertStatus(302)->assertSessionDoesntHaveErrors('channels.0.id');
});

it('stores a new shipping method', function () {
    $data = [
        'name' => ['en' => 'My shipping method'],
        'description' => ['en' => 'My shipping method'],
        'slug' => 'my-shipping-method',
        'channels' => [
            ['id' => defaultChannel()->id, 'price' => 123],
        ],
    ];

    $this->assertDatabaseCount('shipping_methods', 0);
    $this->assertDatabaseCount('channel_shipping_method', 0);

    $this->post('/cp/shipping-methods', $data)
        ->assertStatus(302)
        ->assertRedirect('/cp/shipping-methods');

    $channels = Arr::pull($data, 'channels');
    $data['name'] = json_encode($data['name']);
    $data['description'] = json_encode($data['description']);

    $this->assertDatabaseCount('shipping_methods', 1);
    $this->assertDatabaseCount('channel_shipping_method', 1);
    $this->assertDatabaseHas('shipping_methods', $data);
    $this->assertDatabaseHas('channel_shipping_method', [
        'channel_id' => $channels[0],
        'shipping_method_id' => ShippingMethod::first()->id,
        'price' => 123,
    ]);
});
