<?php

use FirstpointCh\Shop\Models\Attribute;
use FirstpointCh\Shop\Models\Channel;
use FirstpointCh\Shop\Models\Product;
use FirstpointCh\Shop\Models\ProductType;
use FirstpointCh\Shop\Models\Variant;
use Inertia\Testing\AssertableInertia;

it('shows the variant create form', function () {
    $product = Product::factory()->create();

    $this->get('/cp/products/'.$product->id.'/variants/create')
        ->assertStatus(200)
        ->assertInertia(fn (AssertableInertia $page) => $page
            ->component('ProductVariant/Form')
            ->where('product.id', $product->id)
        );
});

it('has default variant data', function () {
    $product = Product::factory()->create();

    $this->get('/cp/products/'.$product->id.'/variants/create')
        ->assertInertia(fn (AssertableInertia $page) => $page
            ->component('ProductVariant/Form')
            ->whereAll([
                'variant.is_active' => false,
                'variant.name' => [config('app.fallback_locale') => null],
                'variant.sku' => null,
                'variant.route_slug' => [config('app.fallback_locale') => null],
                'variant.track_stock' => false,
                'variant.stock' => null,
            ])
        );
});

it('has default attributes based on product type', function () {
    $productType = ProductType::factory()->create();
    $attribute = Attribute::factory()->create();

    Attribute::factory()->create(); // This attribute should not be added by default

    $productType->variantAttributes()->attach($attribute, ['type' => 'variant']);

    $product = Product::factory()->create([
        'product_type_id' => $productType->id,
    ]);

    $this->get('/cp/products/'.$product->id.'/variants/create')
        ->assertInertia(fn (AssertableInertia $page) => $page
            ->component('ProductVariant/Form')
            ->has('variant.base_attributes', 1)
            ->whereAll([
                'variant.base_attributes.0.attribute_id' => $attribute->id,
                'variant.base_attributes.0.name' => $attribute->name,
            ])
        );
});

// TODO: Test that the default attribute value is propertly set according to the attribute type

it('has required fields', function () {
    $product = Product::factory()->create();

    $this->post('/cp/products/'.$product->id.'/variants', [
        'base_attributes' => [[]],
        'extra_attributes' => [[]],
        'channels' => [[]],
    ])->assertSessionHasErrors([
        'name',
        'sku',
        'route_slug',
        'track_stock',
        'base_attributes.0.attribute_id',
        'extra_attributes.0.attribute_id',
        'channels.0.id',
    ]);
});

test('track_stock must be boolean', function () {
    $product = Product::factory()->create();

    $this->post('/cp/products/'.$product->id.'/variants', ['track_stock' => 'test'])
        ->assertSessionHasErrors(['track_stock']);

    $this->post('/cp/products/'.$product->id.'/variants', ['track_stock' => true])
        ->assertSessionDoesntHaveErrors(['track_stock']);
});

test('each attribute must exist', function () {
    $product = Product::factory()->create();
    $attribute = Attribute::factory()->create();

    $this->post('/cp/products/'.$product->id.'/variants', [
        'base_attributes' => [
            ['attribute_id' => $attribute->id],
            ['attribute_id' => 5],
        ],
        'extra_attributes' => [
            ['attribute_id' => $attribute->id],
            ['attribute_id' => 5],
        ],
    ])
    ->assertSessionDoesntHaveErrors(['base_attributes.0.attribute_id', 'extra_attributes.0.attribute_id'])
    ->assertSessionHasErrors(['base_attributes.1.attribute_id', 'extra_attributes.1.attribute_id']);
});

test('each attribute value must exist', function () {
    $product = Product::factory()->create();
    $attribute = Attribute::factory()->create();
    $value = $attribute->options()->create(['value' => 'test']);

    $this->post('/cp/products/'.$product->id.'/variants', [
        'base_attributes' => [
            ['value_option_' => $value->id],
            ['value_option_id' => 14],
        ],
        'extra_attributes' => [
            ['value_option_id' => $value->id],
            ['value_option_id' => 14],
        ],
    ])
    ->assertSessionDoesntHaveErrors(['base_attributes.0.value_option_id', 'extra_attributes.0.value_option_id'])
    ->assertSessionHasErrors(['base_attributes.1.value_option_id', 'extra_attributes.1.value_option_id']);
});

test('each channel must exist', function () {
    $product = Product::factory()->create();
    $channel = Channel::factory()->create();

    $this->post('/cp/products/'.$product->id.'/variants', [
        'channels' => [
            ['id' => $channel->id],
            ['id' => 5],
        ],
    ])
    ->assertSessionDoesntHaveErrors(['channels.0.id'])
    ->assertSessionHasErrors(['channels.1.id']);
});

test('price is required for every active channels', function () {
    $product = Product::factory()->create();
    $channel = Channel::factory()->create();

    $this->post('/cp/products/'.$product->id.'/variants', [
        'channels' => [
            ['id' => $channel->id, 'is_active' => 1, 'price' => null],
            ['id' => $channel->id, 'is_active' => 0, 'price' => null],
        ],
    ])
    ->assertSessionHasErrors(['channels.0.price'])
    ->assertSessionDoesntHaveErrors(['channels.1.price']);
});

it('stores a new product variant', function () {
    $this->withoutExceptionHandling();
    $product = Product::factory()->create();
    $baseAttribute = Attribute::factory()->create();
    $baseAttributeValue = $baseAttribute->options()->create(['value' => 'test']);
    $extraAttribute = Attribute::factory()->create();
    $extraAttributeValue = $extraAttribute->options()->create(['value' => 'test']);
    $channel = Channel::factory()->create();

    $this->post('/cp/products/'.$product->id.'/variants', [
        'is_active' => false,
        'name' => ['en' => 'Test'],
        'sku' => 'test',
        'route_slug' => ['en' => 'test'],
        'track_stock' => false,
        'stock' => null,
        'base_attributes' => [
            ['attribute_id' => $baseAttribute->id, 'value_option_id' => $baseAttributeValue->id],
        ],
        'extra_attributes' => [
            ['attribute_id' => $extraAttribute->id, 'value_option_id' => $extraAttributeValue->id],
        ],
        'channels' => [
            ['id' => $channel->id, 'is_active' => 1, 'price' => 129],
        ],
    ])
    ->assertStatus(302)
    ->assertRedirect('/cp/products/'.$product->id.'/variants');

    $this->assertDatabaseHas('variants', [
        'product_id' => $product->id,
        'is_active' => false,
        'name' => json_encode(['en' => 'Test']),
        'sku' => 'test',
        'track_stock' => false,
        'stock' => null,
        'route_slug' => json_encode(['en' => 'test']),
    ]);

    $variant = Variant::first();

    $this->assertCount(2, $variant->attributes);
    $this->assertEquals($baseAttributeValue->id, $variant->attributes->first()->pivot->value_option_id);
    $this->assertEquals($extraAttributeValue->id, $variant->attributes->last()->pivot->value_option_id);
    $this->assertCount(1, $variant->channels);
    $this->assertEquals(129, $variant->channels->first()->pivot->price);
});
