<?php

namespace FirstpointCh\Shop\Models;

use FirstpointCh\Shop\Exceptions\ProductIsNotAvailable;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class Cart extends Model
{
    use HasFactory;

    protected $guarded = [];

    public $incrementing = false;

    protected $keyType = 'string';

    protected $casts = [
        'is_active' => 'boolean',
    ];

    public $totalTax;

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            if (empty($model->id)) {
                $model->id = Str::orderedUuid()->toString();
            }
        });
    }

    public function channel()
    {
        return $this->belongsTo(Channel::class);
    }

    public function items()
    {
        return $this->hasMany(CartItem::class);
    }

    public function taxes()
    {
        return $this->hasMany(CartTax::class)
            ->with('taxRule')
            ->addSelect([
                'tax_amount' => TaxRule::query()
                    ->selectRaw('cart_taxes.taxable_amount * tax_rules.rate / 100')
                    ->whereColumn('cart_taxes.tax_rule_id', 'tax_rules.id'),
            ]);
    }

    public function add(int $id = null, string $sku = null, int $quantity = 1)
    {
        if (! empty($id)) {
            $variant = Variant::withPrice()->find($id);
        } elseif (! empty($sku)) {
            $variant = Variant::withPrice()->where('sku', $sku)->first();
        } else {
            throw new \Exception('No id or sku provided.');
        }

        if (is_null($variant) || ! $variant->is_active || $variant->product->status !== 'published') {
            throw new ProductIsNotAvailable('The product is not available.');
        }

        if (empty($variant->price)) {
            throw new ProductIsNotAvailable('This product is not available in this channel.');
        }

        if (! $this->id) {
            $this->save();
            $this->refresh();

            session(['shop::cart' => $this->id]);
        }

        $item = $this->items()
            ->where('orderable_id', $variant->id)
            ->where('orderable_type', get_class($variant))
            ->first();

        if (! empty($item)) {
            $item->increment('quantity', $quantity);
        } else {
            $item = $this->items()->create([
                'name' => $variant->product->name,
                'orderable_id' => $variant->id,
                'orderable_type' => get_class($variant),
                'quantity' => $quantity,
                'meta' => null,
            ]);
        }

        return $item;
    }

    public function getSubtotal()
    {
        return $this->items->sum('price');
    }

    public function getTotal()
    {
        $subtotal = $this->getSubTotal();
        $taxes = $this->taxes->excluded()->total();

        // TODO: add shipping

        return $subtotal + $taxes;
    }

    public function reset()
    {
        $this->update([
            'is_active' => false,
        ]);

        session()->forget('shop::cart');

        app('shop::cart')->reset();
    }
}
