<?php

namespace FirstpointCh\Shop\Http\Controllers\Cp;

use FirstpointCh\Shop\Http\Controllers\Controller;
use FirstpointCh\Shop\Http\Requests\DatatableRequest;
use FirstpointCh\Shop\Models\Channel;
use FirstpointCh\Shop\Models\ShippingMethod;
use Illuminate\Support\Arr;
use Inertia\Inertia;

class ShippingMethodController extends Controller
{
    public function index(DatatableRequest $datatable)
    {
        return Inertia::render('ShippingMethod/Index', [
            'shippingMethods' => $datatable
                ->searchable(['name', 'description', 'slug'])
                ->performQuery(
                    ShippingMethod::query()
                ),
            'datatableState' => $datatable->getState(),
        ]);
    }

    public function create()
    {
        return Inertia::render('ShippingMethod/Form', [
            'channels' => Channel::all(),
        ]);
    }

    public function store()
    {
        $data = request()->validate([
            'name' => 'required|array',
            'description' => 'required|array',
            'slug' => 'required|max:255',
            'channels' => 'nullable|array',
            'channels.*.id' => 'required|exists:channels,id',
            'channels.*.price' => 'nullable',
        ]);

        $channels = collect(Arr::pull($data, 'channels'))
            ->keyBy('id')
            ->map(fn ($item) => ['price' => $item['price']])
            ->toArray();

        $shippingMethod = ShippingMethod::create($data);

        $shippingMethod->channels()->attach($channels);

        return redirect('/cp/shipping-methods');
    }

    public function edit(ShippingMethod $shippingMethod)
    {
        return Inertia::render('ShippingMethod/Form', [
            'shippingMethod' => $shippingMethod->in('*')->load('channels'),
            'channels' => Channel::all(),
        ]);
    }

    public function update(ShippingMethod $shippingMethod)
    {
        $data = request()->validate([
            'name' => 'required|array',
            'description' => 'required|array',
            'slug' => 'required|max:255',
            'channels' => 'nullable|array',
            'channels.*.id' => 'required|exists:channels,id',
            'channels.*.price' => 'nullable',
        ]);

        $channels = collect(Arr::pull($data, 'channels'))
            ->keyBy('id')
            ->map(fn ($item) => ['price' => $item['price']])
            ->toArray();

        $shippingMethod->update($data);

        $shippingMethod->channels()->sync($channels);

        return redirect('/cp/shipping-methods');
    }
}
