<?php

namespace FirstpointCh\Shop\Http\Controllers\Cp;

use FirstpointCh\Shop\Data\Cp\DefaultVariantAttributeData;
use FirstpointCh\Shop\Data\Cp\ProductVariantFormData;
use FirstpointCh\Shop\Data\Cp\VariantAttributeData;
use FirstpointCh\Shop\Http\Controllers\Controller;
use FirstpointCh\Shop\Models\Attribute;
use FirstpointCh\Shop\Models\Channel;
use FirstpointCh\Shop\Models\Product;
use FirstpointCh\Shop\Models\TaxRule;
use FirstpointCh\Shop\Models\Variant;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class ProductVariantController extends Controller
{
    public function create(Product $product)
    {
        $variant = new Variant([
            'is_active' => false,
            'name' => [config('app.fallback_locale') => null],
            'sku' => null,
            'route_slug' => [config('app.fallback_locale') => null],
            'is_shippable' => true,
            'weight_grams' => null,
            'track_stock' => false,
            'stock' => null,
            'attributes' => [],
            'channels' => [],
            'base_attributes' => [],
            'extra_attributes' => [],
        ]);

        if (! empty($product->productType)) {
            $variant->setAttribute('base_attributes', DefaultVariantAttributeData::collection(
                $product->productType->variantAttributes
            ));
        }

        return Inertia::render('ProductVariant/Form', [
            'product' => $product,
            'variant' => $variant->in('*'),
            'attributes' => Attribute::all(),
            'channels' => Channel::all(),
            'taxRules' => TaxRule::all(),
            'previewUrlFormat' => config('shop.cp.preview_url_format'),
        ]);
    }

    public function store(Product $product, ProductVariantFormData $data)
    {
        DB::transaction(function () use ($product, $data) {
            $variant = $product->variants()->create($data->getVariantData());
            $variant->attributes()->sync($data->getAttributes());
            $variant->channels()->sync($data->getChannels($product->id));
        });

        return redirect()->route('cp.products.show', [
            'product' => $product,
            'tab' => 'variants',
        ])->with('toast', [
            'title' => 'Variante créé',
            'message' => 'La variante a bien été créé',
            'type' => 'success',
        ]);
    }

    public function edit(Product $product, Variant $variant)
    {
        $baseAttributeIds = $product->productType->variantAttributes->pluck('id')->toArray();

        if (! empty($product->productType)) {
            $baseAttributes = $variant->attributes()->whereIn('attributes.id', $baseAttributeIds)->get();

            $variant->setAttribute('base_attributes', $product->productType->variantAttributes->map(function ($attribute) use ($baseAttributes) {
                return [
                    'attribute_id' => $attribute->id,
                    'name' => $attribute->name,
                    'type' => $attribute->type,
                    'options' => $attribute->options->map->in('*'),
                    'value_option_id' => $baseAttributes->firstWhere('id', $attribute->id)?->pivot->value_option_id,
                    'value_translated' => $baseAttributes->firstWhere('id', $attribute->id)?->pivot->in('*')->value_translated ?? [config('app.fallback_locale') => null],
                    'value_string' => $baseAttributes->firstWhere('id', $attribute->id)?->pivot->value_string,
                    'value_integer' => $baseAttributes->firstWhere('id', $attribute->id)?->pivot->value_integer,
                ];
            }));
        }

        $variant->setAttribute('extra_attributes', VariantAttributeData::collection(
            $variant->attributes()->whereNotIn('attributes.id', $baseAttributeIds)->get()
        ));

        return Inertia::render('ProductVariant/Form', [
            'product' => $product,
            'variant' => $variant->load('channels')->in('*'),
            'attributes' => Attribute::all(),
            'channels' => Channel::all(),
            'taxRules' => TaxRule::all(),
            'previewUrlFormat' => config('shop.cp.preview_url_format'),
        ]);
    }

    public function update(Product $product, Variant $variant, ProductVariantFormData $data)
    {
        DB::transaction(function () use ($variant, $data) {
            $variant->update($data->getVariantData());
            $variant->attributes()->sync($data->getAttributes());
            $variant->channels()->sync($data->getChannels($variant->product_id));
        });

        return redirect()->route('cp.products.show', [
            'product' => $product,
            'tab' => 'variants',
        ])->with('toast', [
            'title' => 'Variante enregistrée',
            'message' => 'La variante a bien été enregistrée',
            'type' => 'success',
        ]);
    }
}
