<?php

namespace FirstpointCh\Shop\Http\Controllers\Cp;

use FirstpointCh\Shop\Data\Cp\DefaultProductAttributeData;
use FirstpointCh\Shop\Data\Cp\ProductAttributeData;
use FirstpointCh\Shop\Data\Cp\ProductFormData;
use FirstpointCh\Shop\Http\Controllers\Controller;
use FirstpointCh\Shop\Http\Requests\DatatableRequest;
use FirstpointCh\Shop\Models\Attribute;
use FirstpointCh\Shop\Models\Product;
use FirstpointCh\Shop\Models\ProductType;
use FirstpointCh\Shop\Models\TaxRule;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;
use Spatie\MediaLibrary\MediaCollections\Models\Media;

class ProductController extends Controller
{
    public function index(DatatableRequest $datatable)
    {
        return Inertia::render('Product/Index', [
            'products' => $datatable
                ->searchable(['name'])
                ->performQuery(
                    Product::with('productType')
                ),
            'datatableState' => $datatable->getState(),
            'productTypes' => ProductType::all(),
        ]);
    }

    public function create()
    {
        $product = new Product([
            'status' => 'draft',
            'name' => [config('app.fallback_locale') => null],
            'product_type_id' => request('productTypeId'),
            'short_description' => [config('app.fallback_locale') => null],
            'description' => [config('app.fallback_locale') => null],
            'base_attributes' => [],
            'extra_attributes' => [],
        ]);

        if (! empty($product->productType)) {
            $product->setAttribute('base_attributes', DefaultProductAttributeData::collection(
                $product->productType->productAttributes
            ));
        }

        return Inertia::render('Product/Form', [
            'product' => $product->in('*'),
            'productTypes' => ProductType::all(),
        ]);
    }

    public function store(ProductFormData $data)
    {
        $product = DB::transaction(function () use ($data) {
            $product = Product::create($data->getProductData());
            $product->attributes()->sync($data->getAttributes());

            return $product;
        });

        return redirect('/cp/products/'.$product->id)->with('toast', [
            'title' => 'Produit créé',
            'message' => 'Le produit a bien été créé',
            'type' => 'success',
        ]);
    }

    public function show(Product $product, string $tab = null)
    {
        $product->load(['attributes']);

        return Inertia::render('Product/Show', [
            'tab' => $tab,
            'product' => $product,
            'variants' => $product->variants()->paginate(10),
            'taxRules' => TaxRule::forProduct($product)->paginate(10),
            'pictures' => $product->getMedia('pictures')->map(function (Media $media) {
                return [
                    'id' => $media->id,
                    'name' => $media->name,
                    'original_url' => $media->getFullUrl(),
                    'thumbnail_url' => $media->getFullUrl('cp_thumbnail'),
                    'alt_text' => $media->getCustomProperty('alt_text'),
                    'size' => $media->size,
                ];
            }),
        ]);
    }

    public function edit(Product $product)
    {
        $baseAttributeIds = $product->productType->productAttributes->pluck('id')->toArray();

        if (! empty($product->productType)) {
            $baseAttributes = $product->attributes()->whereIn('attributes.id', $baseAttributeIds)->get();

            $product->setAttribute('base_attributes', $product->productType->productAttributes->map(function ($attribute) use ($baseAttributes) {
                return [
                    'attribute_id' => $attribute->id,
                    'name' => $attribute->name,
                    'type' => $attribute->type,
                    'options' => $attribute->options->map->in('*'),
                    'value_option_id' => $baseAttributes->firstWhere('id', $attribute->id)?->pivot->value_option_id,
                    'value_translated' => $baseAttributes->firstWhere('id', $attribute->id)?->pivot->in('*')->value_translated ?? [config('app.fallback_locale') => null],
                    'value_string' => $baseAttributes->firstWhere('id', $attribute->id)?->pivot->value_string,
                    'value_integer' => $baseAttributes->firstWhere('id', $attribute->id)?->pivot->value_integer,
                ];
            }));
        }

        $product->setAttribute('extra_attributes', ProductAttributeData::collection(
            $product->attributes()->whereNotIn('attributes.id', $baseAttributeIds)->get()
        ));

        return Inertia::render('Product/Form', [
            'product' => $product->in('*'),
            'productTypes' => ProductType::all(),
            'productTypeAttributes' => $product->productType->productAttributes->map->only('id', 'name', 'type'),
            'attributes' => Attribute::all(),
        ]);
    }

    public function update(Product $product, ProductFormData $data)
    {
        DB::transaction(function () use ($product, $data) {
            $product->update($data->getProductData());
            $product->attributes()->sync($data->getAttributes());
        });

        return redirect('/cp/products/'.$product->id)->with('toast', [
            'title' => 'Produit enregistré',
            'message' => 'Le produit a bien été enregistré',
            'type' => 'success',
        ]);
    }

    public function updateStatus(Product $product)
    {
        $product->update(request()->validate([
            'status' => 'required|in:draft,published',
        ]));

        return back()->with('toast', [
            'title' => 'Produit mis à jour',
            'message' => 'Le produit a bien été '.['draft' => 'désactivé', 'published' => 'publié'][$product->status],
            'type' => 'success',
        ]);
    }

    public function delete(Product $product)
    {
        $product->delete();

        return redirect('/cp/products')->with('toast', [
            'title' => 'Produit supprimé',
            'message' => 'Le produit a bien été supprimé',
            'type' => 'success',
        ]);
    }
}
