<?php

namespace FirstpointCh\Shop;

use Exception;
use FirstpointCh\Shop\Data\Front\CheckoutData;
use FirstpointCh\Shop\Events\OrderPlaced;
use FirstpointCh\Shop\Models\CartItem;
use FirstpointCh\Shop\Models\Order;
use FirstpointCh\Shop\Models\ShippingMethod;
use FirstpointCh\Shop\Order\Enums\OrderItemStatus;
use FirstpointCh\Shop\Order\Enums\OrderStatus;
use FirstpointCh\Shop\Payment\CreatePaymentIntentAction;
use FirstpointCh\Shop\Payment\Gateway;

class Checkout
{
    public CheckoutData $data;

    public function __construct()
    {
        if (! session()->has('shop::checkout')) {
            $this->reset();
        } else {
            $this->data = CheckoutData::from(
                session('shop::checkout')
            );
        }
    }

    public function reset()
    {
        session([
            'shop::checkout' => CheckoutData::empty(),
        ]);

        $this->data = CheckoutData::from(
            session('shop::checkout')
        );
    }

    public function update(array $data)
    {
        foreach ($data as $key => $value) {
            if (! property_exists($this->data, $key)) {
                throw new Exception("Undefined property Checkout::\${$key}");
            }

            $this->data->{$key} = $value;
        }

        session([
            'shop::checkout' => CheckoutData::from($this->data)->toArray(),
        ]);
    }

    public function placeOrder(): Order
    {
        // TODO: Vérifier que les données ainsi que le panier soient valides
        // $this->validate();

        $order = $this->getOrder();

        $order->fill([
            'customer_id' => customer()->id,
            'first_name' => $this->data->first_name,
            'last_name' => $this->data->last_name,
            'email' => customer()->email,
            'company' => $this->data->company,
            'street' => $this->data->street,
            'street_number' => $this->data->street_number,
            'zip' => $this->data->zip,
            'city' => $this->data->city,
            'country' => $this->data->country,
            'use_billing_address' => false,
            'shipping_price' => ShippingMethod::with('channels')
                ->firstWhere('slug', $this->data->shipping_method)
                ->channels
                ->where('id', shop()->getChannel()->id)
                ->first()
                ->pivot
                ->price,
            'amount' => cart()->getTotal(),
            'currency' => cart()->channel->currency,
            'status' => OrderStatus::Pending,
            'payment_method' => $this->data->payment_method,
            'shipping_method' => $this->data->shipping_method,
            'channel_id' => channel()->id,
        ])->save();

        cart()->loadMissing(['items.orderable.product' => fn ($query) => $query->withTotalTaxRate('excluded')]);

        cart()->items->each(fn (CartItem $item) => $order->items()->updateOrCreate([
            'sku' => $item->orderable->sku,
        ], [
            'name' => $item->name,
            'quantity' => $item->quantity,
            'unit_price' => $item->orderable->price,
            'total_tax_rate' => $item->orderable->product->total_tax_rate, // TODO: Test
            'status' => OrderItemStatus::Pending,
        ]));

        $order->items()->whereNotIn('sku', cart()->items->pluck('sku'))->delete();

        $order->taxes()->delete();

        foreach (cart()->taxes as $tax) {
            $order->taxes()->create([
                'name' => $tax->taxRule->name,
                'tax_included' => $tax->taxRule->tax_included,
                'taxable_amount' => $tax->taxable_amount,
                'rate' => $tax->taxRule->rate,
                'tax_amount' => $tax->taxRule->calculate($tax->taxable_amount),
            ]);
        }

        OrderPlaced::dispatch($order);

        $this->update([
            'order_id' => $order->id,
        ]);

        return $order;
    }

    public function getOrder(): Order
    {
        $order = Order::where('status', OrderStatus::Pending)->find(checkout()->data->order_id);

        return $order ?? new Order();
    }

    public function payment($successUrl = null, $errorUrl = null, $cancelUrl = null)
    {
        $this->update([
            'successUrl' => $successUrl,
            'errorUrl' => $errorUrl,
            'cancelUrl' => $cancelUrl,
        ]);

        $payment = CreatePaymentIntentAction::run(
            customer()->orders()->find($this->data->order_id)
        );

        return Gateway::make($payment->gateway)->init($payment);
    }
}
