<?php

namespace FirstpointCh\Shop\Catalog\QueryBuilders;

use FirstpointCh\Shop\Models\Attribute;
use FirstpointCh\Shop\Models\Price;
use Illuminate\Database\Eloquent\Builder;

class VariantQueryBuilder extends Builder
{
    public function withPrice()
    {
        return $this->addSelect(['price' => Price::select('price')
            ->whereColumn('variant_id', 'variants.id')
            ->limit(1),
        ]);
    }

    public function active()
    {
        return $this->where('is_active', true);
    }

    public function whereAttributeIn(string $type, string $attribute, array $values, ?string $locale = null)
    {
        return $this->whereHas('attributes', function ($query) use ($type, $attribute, $values, $locale) {
            $query->where('attributes.key', $attribute);

            foreach ($values as $value) {
                if ($type === 'text') {
                    $query->orWhere('attribute_variant.value_string', $value);
                } elseif ($type === 'translated') {
                    $query->orWhere("attribute_variant.value_translated->{$locale}", $value);
                } elseif ($type === 'integer') {
                    $query->orWhere('attribute_variant.value_integer', $value);
                } elseif ($type === 'options') {
                    $query->orWhere('attribute_variant.value_option_id', $value);
                }
            }
        });
    }

    public function wherePriceBetween($min, $max)
    {
        return $this->whereHas('channels', function ($query) use ($min, $max) {
            $query
                ->when(shop()->isChannelAware(), fn ($query) => $query->where('channel_id', shop()->getChannel()->id))
                ->when($min, fn ($query) => $query->where('price', '>=', $min))
                ->when($max, fn ($query) => $query->where('price', '<=', $max));
        });
    }

    public function whereAttribute(Attribute $attribute, $value)
    {
        if ($attribute->type === 'text') {
            return $this->whereHas('attributes', function ($query) use ($attribute, $value) {
                $query->where('attributes.key', $attribute->key)
                    ->where('attribute_variant.value_string', $value);
            });
        }

        if ($attribute->type === 'translated') {
            return $this->whereHas('attributes', function ($query) use ($attribute, $value) {
                $query->where('attributes.key', $attribute->key)
                    ->where('attribute_variant.value_translated->'.app()->getLocale(), $value);
            });
        }

        if ($attribute->type === 'integer') {
            return $this->whereHas('attributes', function ($query) use ($attribute, $value) {
                $query->where('attributes.key', $attribute->key)
                    ->where('attribute_variant.value_integer', $value);
            });
        }

        if ($attribute->type === 'options') {
            return $this->whereHas('attributes', function ($query) use ($attribute, $value) {
                $query->where('attributes.key', $attribute->key)
                    ->where('attribute_variant.value_option_id', $value);
            });
        }
    }
}
