<?php

namespace FirstpointCh\LunarSwisspost\Http\Barcode\Builder;

use FirstpointCh\LunarSwisspost\Http\Barcode\BarcodeClient;

class BarcodeBuilder
{
    protected $params = [];
    protected $barcodeClient;

	protected $trackingNumber;
	protected $label;
	protected $trackingUrl;

	protected $errors;

    private function __construct(BarcodeClient $barcodeClient)
    {
        $this->barcodeClient = $barcodeClient;

		$this->params = [
			'language' => in_array(app()->getLocale(), ['de', 'fr', 'it', 'en']) ? strtoupper(app()->getLocale()) : 'FR',
			'customer' => [
				'name1' => config('swisspost.customer.name1'),
				'street' => config('swisspost.customer.street'),
				'zip' => config('swisspost.customer.zip'),
				'city' => config('swisspost.customer.city'),
				'country' => config('swisspost.customer.country'),
			],
			'labelDefinition' => [
				'labelLayout' => 'a5',
				'printAddresses' => 'RECIPIENT_AND_CUSTOMER',
				'imageFileType' => 'PNG',
				'imageResolution' => 300,
			],
			'item' => [
				'recipient' => [],
				'attributes' => [
					'przl' => ['pri']
				]
			],
		];
    }

    /**
     * Static method to initialize the BarcodeBuilder.
     *
     * @return self
     */
    public static function build()
    {
        $barcodeClient = app('swisspost.barcode');
		
        return new self($barcodeClient);
    }

	public function przl(array $przl)
	{
		$this->params['item']['attributes']['przl'] = $przl;

		return $this;
	}

	public function license(string $license)
	{
		$this->params['frankingLicense'] = $license;

		return $this;
	}

	public function language(string $language)
	{
		$this->params['language'] = strtoupper($language);

		return $this;
	}

    public function customer(
		string $name1,
		string $street,
		string $zip,
		string $city,
		string $name2 = null,
		string $country = null
	) {
        $this->params['customer'] = array_merge(
			$this->params['customer'],
			collect(get_defined_vars())->filter()->all()
		);

		return $this;
    }

	public function recipient(
		string $name1,
		string $city,
		string $street,
		string $zip,
		string $title = null,
		string $firstName = null,
		string $name2 = null,
		string $name3 = null,
		string $addressSuffix = null,
		string $houseNo = null,
		string $country = null
	) {	
		$this->params['item']['recipient'] = array_merge(
			$this->params['item']['recipient'],
			collect(get_defined_vars())->filter()->all()
		);

		return $this;
	}

    public function layout(string $format)
    {
        $this->params['labelDefinition']['labelLayout'] = $format;
        return $this;
    }

    public function setOption(string $key, $value)
    {
        $this->params[$key] = $value;
        return $this;
    }

    public function create()
    {
		$this->params['frankingLicense'] = in_array('aplus', $this->params['item']['attributes']['przl']) ? config('swisspost.letter_license') : config('swisspost.parcel_license');
		$this->params['customer']['domicilePostOffice'] = $this->params['customer']['zip'] . ' ' . $this->params['customer']['city'];
		
        $response = $this->barcodeClient->executeGenerateBarcode($this->params);

		if (array_key_exists('errors', $response)) {
			if (is_array($response['errors'])) {
				$this->errors = array_map(function($error) {
					if (is_array($error) && array_key_exists('message', $error)) {
						return $error['message'];
					}

					return $error;
				}, $response['errors']);
			} elseif (is_string($response['errors'])) {
				$this->errors = [$response['errors']];
			}

			return $this;
		}

		$this->trackingNumber = $response['identCode'];
		$this->label = $response['label'][0];
		$this->trackingUrl = 'https://www.post.ch/swisspost-tracking?formattedParcelCodes='.$this->trackingNumber;

		return $this;
    }

	public function hasErrors()
	{
		return !empty($this->errors);
	}

	public function getErrors()
	{
		return $this->errors;
	}

	public function getTrackingNumber()
	{
		return $this->trackingNumber;
	}

	public function getLabel()
	{
		return $this->label;
	}

	public function getTrackingUrl()
	{
		return $this->trackingUrl;
	}
}