
# Lunar SwissPost Integration

This plugin integrates Swiss Post Digital Commerce API with Lunar, allowing you to generate shipping labels and track shipments directly from your Lunar-powered e-commerce platform.

## Features

- Generate shipping labels for Swiss Post
- Automatically update order tracking information
- Support for both letter and parcel shipments
- Customizable shipping options
- Integration with Lunar's order management system

## Installation


1. Require the package via Composer:

```bash
composer require firstpoint-ch/lunar-swisspost
```

2. The package will automatically register its service provider.


3. Run the install command:

```bash
php artisan lunar-swisspost:install
```

This command will:
- Publish the configuration file
- Publish the migrations
- Publish the views
- Publish the translations
- Run the migrations

Alternatively, you can perform these steps manually:

- Publish the configuration file:
  ```bash
  php artisan vendor:publish --tag="lunar-swisspost-config"
  ```

- Publish the migrations:
  ```bash
  php artisan vendor:publish --tag="lunar-swisspost-migrations"
  ```

- Publish the views:
  ```bash
  php artisan vendor:publish --tag="lunar-swisspost-views"
  ```

- Publish the translations:
  ```bash
  php artisan vendor:publish --tag="lunar-swisspost-translations"
  ```

- Run the migrations:
  ```bash
  php artisan migrate
  ```

## Configuration

After installation, you can find the configuration file at `config/swisspost.php`. You must set your credentials using the following environment variables:

```
SWISSPOST_CLIENT_ID=your_client_id
SWISSPOST_CLIENT_SECRET=your_client_secret

SWISSPOST_PARCEL_LICENSE=your_parcel_license
SWISSPOST_LETTER_LICENSE=your_letter_license
```

## Usage

The plugin adds a new widget to the Lunar order management page. You can find it in the footer of the order details page.

## Bulk Generation

This package also supports bulk generation of shipping labels. You can generate labels for multiple orders at once using the out-of-the-box Bulk Action in the order index page.

## Automatic order status update

You can automatically update the order status in Lunar after generating a shipping label.

This is done by setting the `post_label_status` in the `swisspost.php` config file to an order status slug.

## Accessing Tracking Information

The plugin will automatically update the order model with the tracking number and tracking URL.

### Multiple Tracking Numbers

As of version 2.0, the package supports multiple tracking numbers per order. Each order can have multiple shipping labels associated with it.

```php
// Access all tracking numbers
$order->trackingNumbers; // Returns a collection of OrderTrackingNumber models

// Access individual tracking numbers
foreach ($order->trackingNumbers as $tracking) {
    $tracking->tracking_number;
    $tracking->tracking_url;
    $tracking->method; // 'pri', 'eco', or 'aplus'
    $tracking->przl; // Array of delivery options
    $tracking->created_at;
}

// Legacy access (still supported for backwards compatibility)
$order->tracking_number; // Returns the first tracking number
$order->tracking_url; // Returns the first tracking URL
```

### Generating Multiple Labels

When dispatching an order, you can specify how many labels to generate:

1. Go to the order detail page
2. Click on "Dispatch"
3. Enable "Generate barcode"
4. Set the "Number of labels" field (1-10)
5. Choose the shipping method and delivery options
6. Submit

All generated tracking numbers will be stored and visible on the order page.

## Customizing the A4 packing slip layout

You can add your logo by defining it's path in the `swisspost.php` config file.

The template can be customized by editing `resources/views/vendor/lunar-swisspost/packing-slip.blade.php`. (Please refer to the installation instructions if the view is not published)

The format of the label within the A4 page can be defined by setting the packing_slip_label_format in the `swisspost.php` config file.

## Generating barcodes elsewhere within or outside of the Lunar ecosystem

You can also generate barcodes outside of the Lunar ecosystem by directly using the `BarcodeBuilder` class.

```php
use FirstpointCh\LunarSwisspost\Http\Barcode\Builder\BarcodeBuilder;

$barcode = new BarcodeBuilder()
	->customer(
		name1: config('swisspost.customer.name1'),
		street: config('swisspost.customer.street'),
		zip: config('swisspost.customer.zip'),
		city: config('swisspost.customer.city'),
		country: config('swisspost.customer.country')
	)
	->recipient(
		name1: 'John Doe',
		street: 'Rue de la Gare 123',
		city: 'Lausanne',
		zip: 1000,
		country: 'ch'
	)
	->layout('a6')
	->pzrl(['pri', 'si'])
	->create();

if ($result->hasErrors()) {
	// Handle errors
} else {
	$trackingNumber = $result->getTrackingNumber();
	$trackingUrl = $result->getTrackingUrl();
	$base64Label = $result->getLabel();
}
```

Additionnally, you can use the `PdfBuilder` class to generate a PDF from the results containing one or multiple labels.

```php
use FirstpointCh\LunarSwisspost\Http\Barcode\Builder\PdfBuilder;

PdfBuilder::build()
	->addLabel($result['base64'], $lunarOrder)
	->download('label.pdf');
```

## Translations

The package includes translations for the following languages:

- English
- French

If you need to add a new translation, you can publish the translations and add your own.

```bash
php artisan vendor:publish --tag=lunar-swisspost-translations
```

## Notes

Please refer to the [Swiss Post Digital Commerce API documentation](https://developer.post.ch/en/digital-commerce-api) for more information on the API.

## License
This plugin is open-sourced software licensed under the MIT license.
