<?php

namespace FirstpointCh\LunarSwisspost\Extensions;

use Filament\Actions\Action;
use Illuminate\Support\Facades\Mail;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Checkbox;
use Filament\Forms\Components\TextInput;
use Filament\Notifications\Notification;
use Filament\Infolists\Components\Section;
use Filament\Forms\Components\CheckboxList;
use Filament\Infolists\Components\Component;
use Filament\Infolists\Components\TextEntry;
use Filament\Infolists\Components\RepeatableEntry;
use Lunar\Admin\Support\Extending\EditPageExtension;
use FirstpointCh\LunarSwisspost\Traits\GeneratesBarcodes;
use FirstpointCh\LunarSwisspost\Http\Barcode\Builder\PdfBuilder;

class ManageOrderExtension extends EditPageExtension
{
    use GeneratesBarcodes;

    protected array $errors = [];

    public function headerActions(array $actions): array
    {
        return [
            ...$actions,

            Action::make('dispatched')
                ->label(__('Dispatch'))
                ->icon('heroicon-o-truck')
                ->color('success')
                ->visible(fn ($record) => $record->status !== 'dispatched')
                ->form([
                    Checkbox::make('generate_barcode')
                        ->label(__('Generate barcode'))
                        ->default(true)
                        ->reactive(),

                    TextInput::make('quantity')
                        ->label(__('Number of labels'))
                        ->numeric()
                        ->default(1)
                        ->minValue(1)
                        ->maxValue(10)
                        ->required(),

                    Select::make('method')
                        ->label(__('Shipping method'))
                        ->options([
                            'pri' => __('Prioritaire'),
                            'eco' => __('Économique'),
                            'aplus' => __('A+')
                        ])
                        ->default('eco')
                        ->required()
                        ->visible(fn ($get) => $get('generate_barcode')),

                    CheckboxList::make('przl')
                        ->label(__('Delivery options'))
                        ->options([
                            'si' => __('Signature')
                        ])
                        ->columns(2)
                        ->visible(fn ($get) => $get('generate_barcode')),
                ])
                ->modalHeading(__('Dispatch Order'))
                ->modalDescription(__('This will mark the order as dispatched. Optionally generate a Swiss Post barcode label.'))
                ->modalSubmitActionLabel(__('Dispatch'))
                ->modalWidth('lg')
                ->visible(fn ($record) => $record->status !== 'dispatched' && $record->status !== 'cancelled')
                ->action(function (array $data, $record) {
                    $order = $record;

                    // Always update status to dispatched
                    $updateData = ['status' => 'dispatched'];

                    $pdf = PdfBuilder::build();
                    $trackingNumbers = [];
                    $hasErrors = false;

                    if ($data['generate_barcode']) {
                        $quantity = $data['quantity'] ?? 1;

                        $results = $this->generateBarcodesForOrder($order, $data, $quantity);

                        foreach ($results as $result) {
                            if (isset($result['errors'])) {
                                $this->errors = array_merge($this->errors, $result['errors']);
                                $hasErrors = true;
                                continue;
                            }

                            $trackingNumbers[] = [
                                'tracking_number' => $result['tracking_number'],
                                'tracking_url' => $result['tracking_url'],
                                'method' => $data['method'],
                                'przl' => $data['przl'] ?? [],
                            ];

                            $pdf->addLabel($result['base64'], $order);
                        }

                        if ($hasErrors && empty($trackingNumbers)) {
                            Notification::make()
                                ->title(__('Error generating barcodes'))
                                ->body(implode(', ', $this->errors))
                                ->danger()
                                ->send();

                            return;
                        }

                        if (!empty($trackingNumbers)) {
                            foreach ($trackingNumbers as $trackingData) {
                                $order->trackingNumbers()->create($trackingData);
                            }

                            $updateData['tracking_number'] = $trackingNumbers[0]['tracking_number'];
                            $updateData['tracking_url'] = $trackingNumbers[0]['tracking_url'];
                        }
                    } else {
                        $quantity = $data['quantity'] ?? 1;

                        for ($i = 0; $i < $quantity; $i++) {
                            $pdf->addLabel(null, $order);
                        }
                    }

                    // Update the order
                    $order->update($updateData);

                    // Send notification based on whether barcode was generated
                    if ($data['generate_barcode'] && !empty($trackingNumbers)) {
                        $labelCount = count($trackingNumbers);
                        Notification::make()
                            ->title(__('Order dispatched successfully'))
                            ->body(__(':count label(s) generated and order marked as dispatched', ['count' => $labelCount]))
                            ->success()
                            ->send();
                    } else {
                        Notification::make()
                            ->title(__('Order dispatched'))
                            ->body(__('Order marked as dispatched'))
                            ->success()
                            ->send();
                    }

                    if ($hasErrors) {
                        Notification::make()
                            ->title(__('Some labels failed'))
                            ->body(implode(', ', array_slice($this->errors, 0, 3)) . (count($this->errors) > 3 ? '...' : ''))
                            ->warning()
                            ->send();
                    }

                    if (config('swisspost.post_label_notification') && $order->user && class_exists(config('swisspost.post_label_notification'))) {
                        $notificationClass = config('swisspost.post_label_notification');

                        if ($order->billingAddress->contact_email) {
                            Mail::to($order->billingAddress->contact_email)->send(new $notificationClass($order));
                        }
                    }

                    if ($pdf) {
                        $filename = $order->reference;
                        if (!empty($trackingNumbers) && count($trackingNumbers) === 1) {
                            $filename .= '_' . $trackingNumbers[0]['tracking_number'];
                        }
                        return $pdf->download($filename . '.pdf');
                    }
                })
                ->modalContentFooter(fn ($action) => view('lunar-swisspost::modal-errors', ['action' => $this]))
        ];
    }

    public function getErrors(): array
    {
        return $this->errors;
    }

    public function extendInfolistAsideSchema(array $schema): array
    {
        return [
            ...$schema,
            Section::make('Tracking')
                ->compact(true)
                ->schema([
                    RepeatableEntry::make('trackingNumbers')
                        ->label(__('Tracking Numbers'))
                        ->schema([
                            TextEntry::make('tracking_number')
                                ->label(__('Tracking number')),

                            TextEntry::make('tracking_url')
                                ->label(__('Tracking URL'))
                                ->url(fn ($record) => $record->tracking_url, true),

                            TextEntry::make('method')
                                ->label(__('Method'))
                                ->formatStateUsing(function ($state) {
                                    return match($state) {
                                        'pri' => __('Prioritaire'),
                                        'eco' => __('Économique'),
                                        'aplus' => __('A+'),
                                        default => $state,
                                    };
                                }),

                            TextEntry::make('created_at')
                                ->label(__('Generated at'))
                                ->dateTime(),
                        ])
                        ->columns(2)
                        ->visible(fn ($record) => $record->trackingNumbers()->exists()),

                    TextEntry::make('tracking_number')
                        ->label(__('Tracking number (legacy)'))
                        ->visible(fn ($record) => $record->tracking_number && !$record->trackingNumbers()->exists()),

                    TextEntry::make('tracking_url')
                        ->label(__('Tracking URL (legacy)'))
                        ->url(fn ($record) => $record->tracking_url, true)
                        ->visible(fn ($record) => $record->tracking_url && !$record->trackingNumbers()->exists()),
                ])
        ];
    }
}
