lunar-swisspost/tests/Unit/Models/OrderTrackingNumberTest.php
<?php
use FirstpointCh\LunarSwisspost\Models\OrderTrackingNumber;
use Illuminate\Support\Facades\Schema;

beforeEach(function () {
    // Ensure the table exists
    if (!Schema::hasTable("lunar_order_tracking_numbers")) {
        Schema::create("lunar_order_tracking_numbers", function ($table) {
            $table->id();
            $table->string("tracking_number");
            $table->string("label_path")->nullable();
            $table->json("przl")->nullable();
            $table->unsignedBigInteger("order_id")->nullable();
            $table->timestamps();
        });
    }
});

it("has correct table name", function () {
    $trackingNumber = new OrderTrackingNumber();

    expect($trackingNumber->getTable())->toBe("lunar_order_tracking_numbers");
});

it("casts przl to array", function () {
    $trackingNumber = new OrderTrackingNumber();

    expect($trackingNumber->getCasts()["przl"])->toBe("array");
});

it("can store przl as array", function () {
    $trackingNumber = new OrderTrackingNumber();
    $trackingNumber->forceFill([
        "tracking_number" => "TEST123",
        "przl" => ["pri", "si"],
    ]);
    $trackingNumber->save();

    $retrieved = OrderTrackingNumber::find($trackingNumber->id);

    expect($retrieved->przl)->toBe(["pri", "si"]);
});

it("can have null przl", function () {
    $trackingNumber = new OrderTrackingNumber();
    $trackingNumber->forceFill([
        "tracking_number" => "TEST123",
        "przl" => null,
    ]);
    $trackingNumber->save();

    $retrieved = OrderTrackingNumber::find($trackingNumber->id);

    expect($retrieved->przl)->toBeNull();
});

it("can store tracking number", function () {
    $trackingNumber = new OrderTrackingNumber();
    $trackingNumber->forceFill([
        "tracking_number" => "TRACK123456",
    ]);
    $trackingNumber->save();

    expect($trackingNumber->tracking_number)->toBe("TRACK123456");
});

it("can store label path", function () {
    $trackingNumber = new OrderTrackingNumber();
    $trackingNumber->forceFill([
        "tracking_number" => "TEST123",
        "label_path" => "labels/swisspost/2025/01/test.png",
    ]);
    $trackingNumber->save();

    expect($trackingNumber->label_path)->toBe(
        "labels/swisspost/2025/01/test.png",
    );
});

it("can have null label path", function () {
    $trackingNumber = new OrderTrackingNumber();
    $trackingNumber->forceFill([
        "tracking_number" => "TEST123",
        "label_path" => null,
    ]);
    $trackingNumber->save();

    expect($trackingNumber->label_path)->toBeNull();
});

it("has order relationship method", function () {
    $trackingNumber = new OrderTrackingNumber();

    // Verify the method exists and has correct return type without
    // actually calling it (which would require Lunar ModelManifest)
    expect(method_exists($trackingNumber, "order"))->toBeTrue();

    $reflection = new ReflectionMethod($trackingNumber, "order");
    expect($reflection->getReturnType()->getName())->toBe(
        \Illuminate\Database\Eloquent\Relations\BelongsTo::class,
    );
});

