lunar-swisspost/tests/Unit/Http/Controllers/DownloadLabelControllerTest.php
<?php
use FirstpointCh\LunarSwisspost\Http\Controllers\DownloadLabelController;
use FirstpointCh\LunarSwisspost\Models\OrderTrackingNumber;
use Illuminate\Support\Facades\Storage;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpKernel\Exception\HttpException;

beforeEach(function () {
    Storage::fake("local");
});

it("returns a download response when label exists", function () {
    // Arrange
    $controller = new DownloadLabelController();
    $trackingNumber = "123456789";
    $filePath = "labels/swisspost/2025/01/15/label.png";

    // Create fake file
    Storage::disk("local")->put($filePath, "fake-image-content");

    // Create the tracking number model using forceFill to bypass mass assignment
    $orderTrackingNumber = new OrderTrackingNumber();
    $orderTrackingNumber->forceFill([
        "tracking_number" => $trackingNumber,
        "label_path" => $filePath,
    ]);

    // Act
    $response = $controller($orderTrackingNumber);

    // Assert
    expect($response)->toBeInstanceOf(BinaryFileResponse::class);
    expect($response->headers->get("Content-Type"))->toBe("image/png");
    expect($response->headers->get("Content-Disposition"))->toContain(
        "attachment",
    );
    expect($response->headers->get("Content-Disposition"))->toContain(
        "123456789.png",
    );
});

it("aborts with 404 when label_path is null", function () {
    // Arrange
    $controller = new DownloadLabelController();

    $orderTrackingNumber = new OrderTrackingNumber();
    $orderTrackingNumber->forceFill([
        "tracking_number" => "123456789",
        "label_path" => null,
    ]);

    // Act & Assert
    try {
        $controller($orderTrackingNumber);
        $this->fail("Expected HttpException was not thrown");
    } catch (HttpException $e) {
        expect($e->getStatusCode())->toBe(404);
    }
});

it("aborts with 404 when label file does not exist", function () {
    // Arrange
    $controller = new DownloadLabelController();
    $filePath = "labels/swisspost/2025/01/15/non-existent.png";

    $orderTrackingNumber = new OrderTrackingNumber();
    $orderTrackingNumber->forceFill([
        "tracking_number" => "123456789",
        "label_path" => $filePath,
    ]);

    // Ensure file does not exist
    Storage::disk("local")->assertMissing($filePath);

    // Act & Assert
    try {
        $controller($orderTrackingNumber);
        $this->fail("Expected HttpException was not thrown");
    } catch (HttpException $e) {
        expect($e->getStatusCode())->toBe(404);
    }
});

it("aborts with 404 when label_path is empty string", function () {
    // Arrange
    $controller = new DownloadLabelController();

    $orderTrackingNumber = new OrderTrackingNumber();
    $orderTrackingNumber->forceFill([
        "tracking_number" => "123456789",
        "label_path" => "",
    ]);

    // Act & Assert
    try {
        $controller($orderTrackingNumber);
        $this->fail("Expected HttpException was not thrown");
    } catch (HttpException $e) {
        expect($e->getStatusCode())->toBe(404);
    }
});

