lunar-swisspost/tests/Unit/Http/Barcode/BarcodeBuilderTest.php
<?php
use FirstpointCh\LunarSwisspost\Http\Barcode\BarcodeClient;
use FirstpointCh\LunarSwisspost\Http\Barcode\Builder\BarcodeBuilder;
use Illuminate\Support\Facades\Config;

beforeEach(function () {
    Config::set("swisspost.customer.name1", "Test Company");
    Config::set("swisspost.customer.street", "Test Street 1");
    Config::set("swisspost.customer.zip", "1000");
    Config::set("swisspost.customer.city", "Lausanne");
    Config::set("swisspost.customer.country", "CH");
    Config::set("swisspost.letter_license", "TEST_LETTER_LICENSE");
    Config::set("swisspost.parcel_license", "TEST_PARCEL_LICENSE");

    // Mock the BarcodeClient
    $mockClient = Mockery::mock(BarcodeClient::class);
    $mockClient->shouldReceive("executeGenerateBarcode")->andReturn([
        "identCode" => "TEST123456789",
        "label" => ["base64encodedstring"],
    ]);

    app()->singleton("swisspost.barcode", fn() => $mockClient);
});

it("builds with default params", function () {
    $builder = BarcodeBuilder::build();

    expect($builder)->toBeInstanceOf(BarcodeBuilder::class);
});

it("sets przl attributes", function () {
    $builder = BarcodeBuilder::build()->przl(["pri", "si"]);

    $reflection = new ReflectionClass($builder);
    $paramsProperty = $reflection->getProperty("params");
    $paramsProperty->setAccessible(true);
    $params = $paramsProperty->getValue($builder);

    expect($params["item"]["attributes"]["przl"])->toBe(["pri", "si"]);
});

it("sets customer data", function () {
    $builder = BarcodeBuilder::build()->customer(
        "Company Name",
        "Main Street",
        "2000",
        "Neuchatel",
        "Company 2",
        "CH",
    );

    $reflection = new ReflectionClass($builder);
    $paramsProperty = $reflection->getProperty("params");
    $paramsProperty->setAccessible(true);
    $params = $paramsProperty->getValue($builder);

    expect($params["customer"]["name1"])->toBe("Company Name");
    expect($params["customer"]["street"])->toBe("Main Street");
    expect($params["customer"]["zip"])->toBe("2000");
    expect($params["customer"]["city"])->toBe("Neuchatel");
    expect($params["customer"]["name2"])->toBe("Company 2");
    expect($params["customer"]["country"])->toBe("CH");
});

it("sets recipient data", function () {
    $builder = BarcodeBuilder::build()->recipient(
        "John Doe",
        "Geneva",
        "Rue du Centre",
        "1201",
        "Mr",
        "John",
        "Doe Inc",
        null,
        "Address Suffix",
        "House 1",
        "CH",
    );

    $reflection = new ReflectionClass($builder);
    $paramsProperty = $reflection->getProperty("params");
    $paramsProperty->setAccessible(true);
    $params = $paramsProperty->getValue($builder);

    expect($params["item"]["recipient"]["name1"])->toBe("John Doe");
    expect($params["item"]["recipient"]["city"])->toBe("Geneva");
    expect($params["item"]["recipient"]["street"])->toBe("Rue du Centre");
    expect($params["item"]["recipient"]["zip"])->toBe("1201");
    expect($params["item"]["recipient"]["title"])->toBe("Mr");
    expect($params["item"]["recipient"]["firstName"])->toBe("John");
    expect($params["item"]["recipient"]["name2"])->toBe("Doe Inc");
    expect($params["item"]["recipient"]["addressSuffix"])->toBe(
        "Address Suffix",
    );
    expect($params["item"]["recipient"]["houseNo"])->toBe("House 1");
    expect($params["item"]["recipient"]["country"])->toBe("CH");
});

it("sets layout", function () {
    $builder = BarcodeBuilder::build()->layout("a4");

    $reflection = new ReflectionClass($builder);
    $paramsProperty = $reflection->getProperty("params");
    $paramsProperty->setAccessible(true);
    $params = $paramsProperty->getValue($builder);

    expect($params["labelDefinition"]["labelLayout"])->toBe("a4");
});

it("sets language", function () {
    $builder = BarcodeBuilder::build()->language("DE");

    $reflection = new ReflectionClass($builder);
    $paramsProperty = $reflection->getProperty("params");
    $paramsProperty->setAccessible(true);
    $params = $paramsProperty->getValue($builder);

    expect($params["language"])->toBe("DE");
});

it("sets custom option", function () {
    $builder = BarcodeBuilder::build()->setOption("customKey", "customValue");

    $reflection = new ReflectionClass($builder);
    $paramsProperty = $reflection->getProperty("params");
    $paramsProperty->setAccessible(true);
    $params = $paramsProperty->getValue($builder);

    expect($params["customKey"])->toBe("customValue");
});

it("creates barcode and returns tracking data", function () {
    $builder = BarcodeBuilder::build()
        ->customer("Test", "Street", "1000", "City")
        ->recipient("John", "City", "Street", "1000")
        ->create();

    expect($builder->getTrackingNumber())->toBe("TEST123456789");
    expect($builder->getLabel())->toBe("base64encodedstring");
    expect($builder->getTrackingUrl())->toBe(
        "https://www.post.ch/swisspost-tracking?formattedParcelCodes=TEST123456789",
    );
    expect($builder->hasErrors())->toBeFalse();
});

it("handles errors from api", function () {
    // Mock client with error response
    $mockClient = Mockery::mock(BarcodeClient::class);
    $mockClient->shouldReceive("executeGenerateBarcode")->andReturn([
        "errors" => [
            ["message" => "Invalid address"],
            ["message" => "Invalid zip code"],
        ],
    ]);

    app()->singleton("swisspost.barcode", fn() => $mockClient);

    $builder = BarcodeBuilder::build()
        ->customer("Test", "Street", "1000", "City")
        ->recipient("John", "City", "Street", "1000")
        ->create();

    expect($builder->hasErrors())->toBeTrue();
    expect($builder->getErrors())->toBe([
        "Invalid address",
        "Invalid zip code",
    ]);
    expect($builder->getTrackingNumber())->toBeNull();
    expect($builder->getLabel())->toBeNull();
});

it("handles string errors from api", function () {
    $mockClient = Mockery::mock(BarcodeClient::class);
    $mockClient->shouldReceive("executeGenerateBarcode")->andReturn([
        "errors" => "Connection timeout",
    ]);

    app()->singleton("swisspost.barcode", fn() => $mockClient);

    $builder = BarcodeBuilder::build()->create();

    expect($builder->hasErrors())->toBeTrue();
    expect($builder->getErrors())->toBe(["Connection timeout"]);
});

it("uses letter license for aplus method", function () {
    $licenseUsed = false;

    $mockClient = Mockery::mock(BarcodeClient::class);
    $mockClient
        ->shouldReceive("executeGenerateBarcode")
        ->with(
            Mockery::on(function ($params) use (&$licenseUsed) {
                $licenseUsed =
                    $params["frankingLicense"] === "TEST_LETTER_LICENSE";
                return $licenseUsed;
            }),
        )
        ->andReturn([
            "identCode" => "TEST123",
            "label" => ["label"],
        ]);

    app()->singleton("swisspost.barcode", fn() => $mockClient);

    BarcodeBuilder::build()
        ->przl(["aplus"])
        ->customer("Test", "Street", "1000", "City")
        ->recipient("John", "City", "Street", "1000")
        ->create();

    expect($licenseUsed)->toBeTrue();
});

it("uses parcel license for non-aplus methods", function () {
    $licenseUsed = false;

    $mockClient = Mockery::mock(BarcodeClient::class);
    $mockClient
        ->shouldReceive("executeGenerateBarcode")
        ->with(
            Mockery::on(function ($params) use (&$licenseUsed) {
                $licenseUsed =
                    $params["frankingLicense"] === "TEST_PARCEL_LICENSE";
                return $licenseUsed;
            }),
        )
        ->andReturn([
            "identCode" => "TEST456",
            "label" => ["label"],
        ]);

    app()->singleton("swisspost.barcode", fn() => $mockClient);

    BarcodeBuilder::build()
        ->przl(["pri", "eco"])
        ->customer("Test", "Street", "1000", "City")
        ->recipient("John", "City", "Street", "1000")
        ->create();

    expect($licenseUsed)->toBeTrue();
});

it("adds domicile post office to customer", function () {
    $domicilePostOfficeAdded = false;

    $mockClient = Mockery::mock(BarcodeClient::class);
    $mockClient
        ->shouldReceive("executeGenerateBarcode")
        ->with(
            Mockery::on(function ($params) use (&$domicilePostOfficeAdded) {
                $domicilePostOfficeAdded =
                    $params["customer"]["domicilePostOffice"] ===
                    "1000 Lausanne";
                return $domicilePostOfficeAdded;
            }),
        )
        ->andReturn([
            "identCode" => "TEST123",
            "label" => ["label"],
        ]);

    app()->singleton("swisspost.barcode", fn() => $mockClient);

    BarcodeBuilder::build()
        ->customer("Test", "Street", "1000", "Lausanne")
        ->recipient("John", "City", "Street", "1000")
        ->create();

    expect($domicilePostOfficeAdded)->toBeTrue();
});

afterEach(function () {
    Mockery::close();
});

