<?php

use FirstpointCh\LunarSwisspost\Http\Auth\Authenticator;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\StreamInterface;

beforeEach(function () {
    Config::set("swisspost.client_id", "test-client-id");
    Config::set("swisspost.client_secret", "test-client-secret");
    Cache::flush();
});

it("retrieves access token from cache when available", function () {
    // Arrange
    Cache::put(
        "swisspost_access_token_dcapi_barcode_read",
        "cached-token",
        3600,
    );

    $authenticator = new Authenticator();

    // Act
    $token = $authenticator->getAccessToken();

    // Assert
    expect($token)->toBe("cached-token");
});

it("requests new token when not cached", function () {
    // Arrange - Create proper mock response
    $mockStream = Mockery::mock(StreamInterface::class);
    $mockStream->shouldReceive("__toString")->andReturn(
        json_encode([
            "access_token" => "new-access-token",
            "expires_in" => 3600,
        ]),
    );

    $mockResponse = Mockery::mock(ResponseInterface::class);
    $mockResponse->shouldReceive("getStatusCode")->andReturn(200);
    $mockResponse->shouldReceive("getBody")->andReturn($mockStream);

    $mockClient = Mockery::mock("GuzzleHttp\Client");
    $mockClient->shouldReceive("post")->once()->andReturn($mockResponse);

    $authenticator = new Authenticator();

    // Inject mock client via reflection
    $reflection = new ReflectionClass($authenticator);
    $clientProperty = $reflection->getProperty("client");
    $clientProperty->setAccessible(true);
    $clientProperty->setValue($authenticator, $mockClient);

    // Act
    $token = $authenticator->getAccessToken();

    // Assert
    expect($token)->toBe("new-access-token");
    expect(Cache::get("swisspost_access_token_dcapi_barcode_read"))->toBe(
        "new-access-token",
    );
});

it("caches token with correct expiration", function () {
    // Arrange
    $mockStream = Mockery::mock(StreamInterface::class);
    $mockStream->shouldReceive("__toString")->andReturn(
        json_encode([
            "access_token" => "cached-token",
            "expires_in" => 7200,
        ]),
    );

    $mockResponse = Mockery::mock(ResponseInterface::class);
    $mockResponse->shouldReceive("getStatusCode")->andReturn(200);
    $mockResponse->shouldReceive("getBody")->andReturn($mockStream);

    $mockClient = Mockery::mock("GuzzleHttp\Client");
    $mockClient->shouldReceive("post")->once()->andReturn($mockResponse);

    $authenticator = new Authenticator();

    $reflection = new ReflectionClass($authenticator);
    $clientProperty = $reflection->getProperty("client");
    $clientProperty->setAccessible(true);
    $clientProperty->setValue($authenticator, $mockClient);

    // Act
    $authenticator->getAccessToken();

    // Assert
    expect(Cache::has("swisspost_access_token_dcapi_barcode_read"))->toBeTrue();
});

it("throws exception when token request fails", function () {
    // Arrange
    $mockStream = Mockery::mock(StreamInterface::class);
    $mockStream
        ->shouldReceive("__toString")
        ->andReturn(json_encode(["error" => "invalid_client"]));

    $mockResponse = Mockery::mock(ResponseInterface::class);
    $mockResponse->shouldReceive("getStatusCode")->andReturn(401);
    $mockResponse->shouldReceive("getBody")->andReturn($mockStream);

    $mockClient = Mockery::mock("GuzzleHttp\Client");
    $mockClient->shouldReceive("post")->once()->andReturn($mockResponse);

    $authenticator = new Authenticator();

    $reflection = new ReflectionClass($authenticator);
    $clientProperty = $reflection->getProperty("client");
    $clientProperty->setAccessible(true);
    $clientProperty->setValue($authenticator, $mockClient);

    // Act & Assert
    $this->expectException(\Exception::class);
    $this->expectExceptionMessage("Failed to retrieve access token");

    $authenticator->getAccessToken();
});

it("uses custom scope when provided", function () {
    // Arrange
    Cache::put("swisspost_access_token_custom_scope", "scoped-token", 3600);

    $authenticator = new Authenticator();

    // Act
    $token = $authenticator->getAccessToken("CUSTOM_SCOPE");

    // Assert
    expect($token)->toBe("scoped-token");
});

afterEach(function () {
    Mockery::close();
    Cache::flush();
});
