<?php

use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Route;
use FirstpointCh\LunarSwisspost\Models\OrderTrackingNumber;
use function Pest\Laravel\get;

beforeEach(function () {
    Storage::fake("local");

    // Ensure routes are registered
    Route::middleware(["web"])
        ->name("lunar-swisspost.")
        ->prefix("lunar-swisspost")
        ->group(function () {
            Route::get(
                "tracking-numbers/{trackingNumber}/label",
                \FirstpointCh\LunarSwisspost\Http\Controllers\DownloadLabelController::class,
            )->name("labels.download");
        });
});

it("can download a label file successfully", function () {
    // Arrange
    $trackingNumber = "123456789";
    $fileName = "label.png";
    $filePath = "labels/swisspost/2025/01/15/" . $fileName;

    // Create a fake file in storage
    Storage::disk("local")->put($filePath, "fake-image-content");

    // Create the tracking number model using forceFill to bypass mass assignment
    // and save to database to get an ID for the route parameter
    $orderTrackingNumber = new OrderTrackingNumber();
    $orderTrackingNumber->forceFill([
        "tracking_number" => $trackingNumber,
        "label_path" => $filePath,
    ]);
    $orderTrackingNumber->save();

    // Act
    $response = get(
        route("lunar-swisspost.labels.download", $orderTrackingNumber),
    );

    // Assert
    $response->assertOk();
    $response->assertHeader("Content-Type", "image/png");
    $response->assertDownload($trackingNumber . ".png");
});

it("returns 404 when label_path is null", function () {
    // Arrange
    $orderTrackingNumber = new OrderTrackingNumber();
    $orderTrackingNumber->forceFill([
        "tracking_number" => "123456789",
        "label_path" => null,
    ]);
    $orderTrackingNumber->save();

    // Act
    $response = get(
        route("lunar-swisspost.labels.download", $orderTrackingNumber),
    );

    // Assert
    $response->assertNotFound();
});

it("returns 404 when label file does not exist in storage", function () {
    // Arrange
    $filePath = "labels/swisspost/2025/01/15/non-existent.png";

    $orderTrackingNumber = new OrderTrackingNumber();
    $orderTrackingNumber->forceFill([
        "tracking_number" => "123456789",
        "label_path" => $filePath,
    ]);
    $orderTrackingNumber->save();

    // Ensure file does not exist
    Storage::disk("local")->assertMissing($filePath);

    // Act
    $response = get(
        route("lunar-swisspost.labels.download", $orderTrackingNumber),
    );

    // Assert
    $response->assertNotFound();
});

it("returns 404 when label_path is empty string", function () {
    // Arrange
    $orderTrackingNumber = new OrderTrackingNumber();
    $orderTrackingNumber->forceFill([
        "tracking_number" => "123456789",
        "label_path" => "",
    ]);
    $orderTrackingNumber->save();

    // Act
    $response = get(
        route("lunar-swisspost.labels.download", $orderTrackingNumber),
    );

    // Assert
    $response->assertNotFound();
});
