<?php

namespace FirstpointCh\LunarSwisspost\Http\Barcode\Builder;

use Lunar\Models\Order;
use TCPDF;

class PdfBuilder
{
	protected $pdf;
	protected $format;

	private function __construct(TCPDF $pdf, string $format)
	{
		$this->pdf = $pdf;
		$this->format = $format;
	}

	public static function build(): self
	{
		$format = strtoupper(config('swisspost.layout', 'a4'));
        $pdf = new TCPDF($format === 'A4' ? 'P' : 'L', 'mm', $format);

        $pdf->SetMargins(0, 0, 0);
        $pdf->SetAutoPageBreak(false);
		$pdf->setPrintHeader(false);
		$pdf->setPrintFooter(false);

		return new self($pdf, $format);
	}

	public function addLabel(string $base64, Order $order = null): self
	{
		$this->pdf->AddPage();

		$img = base64_decode(preg_replace('#^data:image/[^;]+;base64,#', '', $base64));

		if ($this->format === 'A4') {
			[$height, $width] = [
				'a5' => [148, 210],
				'a6' => [105 * 0.8, 148 * 0.8],
				'a7' => [74, 105],
			][config('swisspost.packing_slip_label_format')];

			$labelWidth = $width;
			$pageWidth = $this->pdf->getPageWidth();
			$xPosition = $pageWidth - $labelWidth;

			$this->pdf->Image(
				file: "@".$img,
				x: $xPosition - 5,
				y: 0,
				w: $labelWidth,
				h: $height
			);

			$this->pdf->SetY($height + 10);
			$this->pdf->SetMargins(10, 10, 10);

            $locale = app()->getLocale();

            app()->setLocale($order->meta['locale'] ?? config('app.fallback_locale'));

			$this->pdf->writeHTML(view('lunar-swisspost::packing-slip', [
				'order' => $order->load('customer'),
			])->render());

            app()->setLocale($locale);
		} else {
			$this->pdf->Image(
				file: "@".$img,
				w: $this->pdf->getPageWidth(),
				h: $this->pdf->getPageHeight()
			);
		}

		return $this;
	}

	public function download($filename)
	{
		return response()->streamDownload(
			function() {
				echo $this->pdf->Output('', 'S');
			},
			$filename,
			['Content-Type' => 'application/pdf']
		);
	}
}
