<?php

namespace FirstpointCh\LunarSwisspost\Extensions;

use Filament\Actions\Action;
use Filament\Forms\Components\Checkbox;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\CheckboxList;
use Filament\Notifications\Notification;
use FirstpointCh\LunarSwisspost\Http\Barcode\Builder\PdfBuilder;
use FirstpointCh\LunarSwisspost\Traits\GeneratesBarcodes;
use Illuminate\Support\Facades\Mail;
use Lunar\Admin\Support\Extending\EditPageExtension;

class ManageOrderExtension extends EditPageExtension
{
    use GeneratesBarcodes;

    protected array $errors = [];

    public function headerActions(array $actions): array
    {
        return [
            ...$actions,

            Action::make('dispatched')
                ->label(__('Dispatch'))
                ->icon('heroicon-o-truck')
                ->color('success')
                ->visible(fn ($record) => $record->status !== 'dispatched')
                ->form([
                    Checkbox::make('generate_barcode')
                        ->label(__('Generate barcode'))
                        ->default(true)
                        ->reactive(),

                    Select::make('method')
                        ->label(__('Shipping method'))
                        ->options([
                            'pri' => __('Prioritaire'),
                            'eco' => __('Économique'),
                            'aplus' => __('A+')
                        ])
                        ->default('eco')
                        ->required()
                        ->visible(fn ($get) => $get('generate_barcode')),

                    CheckboxList::make('przl')
                        ->label(__('Delivery options'))
                        ->options([
                            'si' => __('Signature')
                        ])
                        ->columns(2)
                        ->visible(fn ($get) => $get('generate_barcode')),
                ])
                ->modalHeading(__('Dispatch Order'))
                ->modalDescription(__('This will mark the order as dispatched. Optionally generate a Swiss Post barcode label.'))
                ->modalSubmitActionLabel(__('Dispatch'))
                ->modalWidth('lg')
                ->visible(fn ($record) => $record->status !== 'dispatched' && $record->status !== 'cancelled')
                ->action(function (array $data, $record) {
                    $order = $record;

                    // Always update status to dispatched
                    $updateData = ['status' => 'dispatched'];

                    $pdf = null;

                    if ($data['generate_barcode']) {
                        $result = $this->generateBarcodeForOrder($order, $data);

                        if (isset($result['errors'])) {
                            $this->errors = $result['errors'];

                            Notification::make()
                                ->title(__('Error generating barcode'))
                                ->body(implode(', ', $result['errors']))
                                ->danger()
                                ->send();

                            return;
                        }

                        // Add tracking information to update data
                        $updateData['tracking_number'] = $result['tracking_number'];
                        $updateData['tracking_url'] = $result['tracking_url'];

                        // Prepare PDF for download
                        $pdf = PdfBuilder::build()
                            ->addLabel($result['base64'], $order);
                    }

                    // Update the order
                    $order->update($updateData);

                    // Send notification based on whether barcode was generated
                    if ($data['generate_barcode'] && !$this->errors) {
                        Notification::make()
                            ->title(__('Order dispatched successfully'))
                            ->body(__('Barcode generated and order marked as dispatched'))
                            ->success()
                            ->send();
                    } else {
                        Notification::make()
                            ->title(__('Order dispatched'))
                            ->body(__('Order marked as dispatched'))
                            ->success()
                            ->send();
                    }

                    if (config('swisspost.post_label_notification') && $order->user && class_exists(config('swisspost.post_label_notification'))) {
                        $notificationClass = config('swisspost.post_label_notification');

                        if ($order->billingAddress->contact_email) {
                            Mail::to($order->billingAddress->contact_email)->send(new $notificationClass($order));
                        }
                    }

                    if ($pdf) {
                        return $pdf->download($order->id . '_' . $order->tracking_number . '.pdf');
                    }
                })
                ->modalContentFooter(fn ($action) => view('lunar-swisspost::modal-errors', ['action' => $this]))
        ];
    }

    public function getErrors(): array
    {
        return $this->errors;
    }
}
