<?php

namespace FirstpointCh\LunarSwisspost\Actions;

use Filament\Forms\Components\Checkbox;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\CheckboxList;
use Filament\Notifications\Notification;
use Filament\Tables\Actions\BulkAction;
use FirstpointCh\LunarSwisspost\Http\Barcode\Builder\PdfBuilder;
use Illuminate\Database\Eloquent\Collection;
use FirstpointCh\LunarSwisspost\Traits\GeneratesBarcodes;
use Illuminate\Support\Facades\Mail;
use Lunar\Models\Order;

class GenerateBarcodesBulkAction extends BulkAction
{
	use GeneratesBarcodes;

	protected array $errors = [];

    protected function setUp(): void
    {
        parent::setUp();

        $this
            ->label(__('Dispatch'))
            ->icon('heroicon-o-truck')
            ->color('success')
            ->action(fn (Collection $records, array $data) => $this->dispatchOrders($records, $data))
            ->form([
                Checkbox::make('generate_barcode')
                    ->label(__('Generate barcode'))
                    ->default(true)
                    ->reactive(),

                Select::make('method')
                    ->label(__('Shipping method'))
                    ->options([
                        'pri' => __('Prioritaire'),
                        'eco' => __('Économique'),
                        'aplus' => __('A+')
                    ])
                    ->default('eco')
                    ->required()
                    ->visible(fn ($get) => $get('generate_barcode')),

                CheckboxList::make('przl')
                    ->label(__('Delivery options'))
                    ->options([
                        'si' => __('Signature')
                    ])
                    ->columns(2)
                    ->visible(fn ($get) => $get('generate_barcode')),
            ])
            ->modalHeading(__('Dispatch'))
            ->modalDescription(__('This will mark the selected orders as dispatched. Optionally generate Swiss Post barcode labels.'))
            ->modalSubmitActionLabel(__('Dispatch'))
            ->modalWidth('lg')
            ->modalContentFooter(fn ($action) => view('lunar-swisspost::modal-errors', ['action' => $action]));
    }

    public function dispatchOrders(Collection $records, array $data)
    {
        $pdf = null;
        $successCount = 0;
        $errorCount = 0;
        $barcodeGenerated = false;

        if ($data['generate_barcode']) {
            $pdf = PdfBuilder::build();
        }

        foreach ($records as $order) {
            // Always update status to dispatched
            $updateData = ['status' => 'dispatched'];

            if ($data['generate_barcode']) {
                $result = $this->generateBarcodeForOrder($order, $data);

                if (isset($result['errors'])) {
                    $this->addErrors($order->id, $result['errors']);
                    $errorCount++;

                    // Still update order status even if barcode generation failed
                    $order->update($updateData);
                    continue;
                }

                // Add tracking information to update data
                $updateData['tracking_number'] = $result['tracking_number'];
                $updateData['tracking_url'] = $result['tracking_url'];

                $pdf->addLabel($result['base64'], $order);
                $barcodeGenerated = true;
            }

            // Update the order
            $order->update($updateData);

            // Send email notification if configured
            if (config('swisspost.post_label_notification') && $order->user && class_exists(config('swisspost.post_label_notification'))) {
                $notificationClass = config('swisspost.post_label_notification');

                if ($order->billingAddress->contact_email) {
                    Mail::to($order->billingAddress->contact_email)->send(new $notificationClass($order));
                }
            }

            $successCount++;
        }

        // Send appropriate notifications
        if (!empty($this->errors)) {
            Notification::make()
                ->title(__('Some orders could not be processed'))
                ->body(implode(', ', array_slice($this->errors, 0, 3)) . (count($this->errors) > 3 ? '...' : ''))
                ->warning()
                ->send();
        }

        if ($successCount > 0) {
            if ($data['generate_barcode'] && $barcodeGenerated) {
                Notification::make()
                    ->title(__('Orders dispatched successfully'))
                    ->body(__('Barcodes generated and orders marked as dispatched'))
                    ->success()
                    ->send();
            } else {
                Notification::make()
                    ->title(__('Orders dispatched'))
                    ->body(__('Orders marked as dispatched'))
                    ->success()
                    ->send();
            }
        }

        // Download PDF if barcodes were generated
        if ($pdf && $barcodeGenerated) {
            return $pdf->download('dispatch_labels_' . now()->format('YmdHis') . '.pdf');
        }
    }

    protected function addErrors(int $orderId, array $errors): void
    {
        $this->errors = array_merge(
            $this->errors,
            array_map(fn ($error) => __("Order") . " {$orderId} : {$error}", $errors)
        );
    }

    public function getErrors(): array
    {
        return $this->errors;
    }
}
