<?php

namespace FirstPoint\LunarSaferpay;

use GuzzleHttp\Client;
use Lunar\Base\DataTransferObjects\PaymentAuthorize;
use Lunar\Base\DataTransferObjects\PaymentCapture;
use Lunar\Base\DataTransferObjects\PaymentRefund;
use Lunar\Models\Cart;
use Lunar\Models\Order;
use Lunar\Models\Transaction;
use Lunar\PaymentTypes\AbstractPayment;

class SaferpayPaymentDriver extends AbstractPayment
{
    protected ?Cart $cart = null;
    protected Client $client;
    protected array $config = [];
    protected ?string $accessToken = null;
    protected ?Order $order = null;

    public function __construct()
    {
        $this->config = config('lunar-saferpay');
        $this->initializeClient();
    }

    protected function initializeClient(): void
    {
        $baseUrl = $this->config['sandbox'] 
            ? 'https://test.saferpay.com/api'
            : 'https://www.saferpay.com/api';

        $this->client = new Client([
            'base_uri' => $baseUrl,
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'Authorization' => 'Basic ' . base64_encode($this->config['username'] . ':' . $this->config['password'])
            ]
        ]);
    }

    public function cart(Cart $cart): self
    {
        $this->cart = $cart;
        return $this;
    }

    public function authorize(): PaymentAuthorize
    {
        try {
            if (!$this->order) {
                if (!$this->order = $this->cart->order) {
                    $this->order = $this->cart->createOrder();
                }
            }

            $payload = [
                'RequestHeader' => [
                    'SpecVersion' => '1.44',
                    'CustomerId' => $this->config['customer_id'],
                    'RequestId' => uniqid(),
                    'RetryIndicator' => 0
                ],
                'TerminalId' => $this->config['terminal_id'],
                'Payment' => [
                    'Amount' => [
                        'Value' => $this->cart->total->value,
                        'CurrencyCode' => $this->cart->currency->code
                    ],
                    'OrderId' => $this->order->id,
                    'Description' => 'Order ' . $this->order->id
                ],
                'ReturnUrl' => [
                    'Url' => route('saferpay.success', ['cart_id' => $this->cart->id]),
                    'NotifyUrl' => route('saferpay.success', ['cart_id' => $this->cart->id])
                ]
            ];

            $response = $this->client->post('Payment/v1/PaymentPage/Initialize', [
                'json' => $payload
            ]);

            $saferpayResponse = new SaferpayPaymentResponse($response);

            // Create transaction
            Transaction::create([
                'order_id' => $this->order->id,
                'type' => 'intent',
                'driver' => 'saferpay',
                'amount' => $this->cart->total->value,
                'reference' => $saferpayResponse->getToken(),
                'status' => 'pending',
                'success' => true,
            ]);

            $this->withData([
                'redirect_url' => $saferpayResponse->getRedirectUrl()
            ]);

            return new PaymentAuthorize(
                success: true,
                message: $saferpayResponse->message(),
                orderId: $this->order->id,
                paymentType: 'saferpay'
            );
        } catch (\Exception $e) {
            return new PaymentAuthorize(
                success: false,
                message: $e->getMessage()
            );
        }
    }

    public function capture(Transaction $transaction, $amount = 0): PaymentCapture
    {
        try {
            $response = $this->client->post('Payment/v1/Transaction/Capture', [
                'json' => [
                    'RequestHeader' => [
                        'SpecVersion' => '1.44',
                        'CustomerId' => $this->config['customer_id'],
                        'RequestId' => uniqid(),
                        'RetryIndicator' => 0
                    ],
                    'TransactionReference' => [
                        'TransactionId' => $transaction->reference
                    ]
                ]
            ]);

            $data = json_decode($response->getBody(), true);

            return new PaymentCapture(
                success: true,
                message: 'Payment captured successfully'
            );
        } catch (\Exception $e) {
            return new PaymentCapture(
                success: false,
                message: $e->getMessage()
            );
        }
    }

    public function refund(Transaction $transaction, int $amount = 0, $notes = null): PaymentRefund
    {
        try {
            $response = $this->client->post('Payment/v1/Transaction/Refund', [
                'json' => [
                    'RequestHeader' => [
                        'SpecVersion' => '1.44',
                        'CustomerId' => $this->config['customer_id'],
                        'RequestId' => uniqid(),
                        'RetryIndicator' => 0
                    ],
                    'TransactionReference' => [
                        'TransactionId' => $transaction->reference
                    ],
                    'Amount' => [
                        'Value' => $amount ?: $transaction->amount,
                        'CurrencyCode' => $transaction->currency->code
                    ]
                ]
            ]);

            $data = json_decode($response->getBody(), true);

            return new PaymentRefund(
                success: true,
                message: 'Refund processed successfully'
            );
        } catch (\Exception $e) {
            return new PaymentRefund(
                success: false,
                message: $e->getMessage()
            );
        }
    }

    public function getPaymentUrl(): ?string
    {
        try {
            if (!$this->order) {
                if (!$this->order = $this->cart->order) {
                    $this->order = $this->cart->createOrder();
                }
            }

            $payload = [
                'RequestHeader' => [
                    'SpecVersion' => '1.44',
                    'CustomerId' => $this->config['customer_id'],
                    'RequestId' => uniqid(),
                    'RetryIndicator' => 0
                ],
                'TerminalId' => $this->config['terminal_id'],
                'Payment' => [
                    'Amount' => [
                        'Value' => $this->cart->total->value,
                        'CurrencyCode' => $this->cart->currency->code
                    ],
                    'OrderId' => $this->order->id,
                    'Description' => 'Order ' . $this->order->id
                ],
                'ReturnUrl' => [
                    'Url' => route('saferpay.success', ['cart_id' => $this->cart->id]),
                    'NotifyUrl' => route('saferpay.success', ['cart_id' => $this->cart->id])
                ]
            ];

            $response = $this->client->post('Payment/v1/PaymentPage/Initialize', [
                'json' => $payload
            ]);

            $saferpayResponse = new SaferpayPaymentResponse($response);
            
            return $saferpayResponse->getRedirectUrl();
        } catch (\Exception $e) {
            report($e);
            return null;
        }
    }
}
