<?php

namespace FirstPoint\LunarPayrexx\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Log;
use Lunar\Models\Cart;
use Lunar\Models\Order;
use Lunar\Models\Transaction;

class PayrexxWebhookController
{
    /**
     * Handle the webhook request
     *
     * @return \Illuminate\Http\Response
     */
    public function handle(Request $request)
    {
        $webhookData = $request->input('transaction', []);

        if (empty($webhookData)) {
            return response('No transaction data found', 400);
        }

        $referenceId = $webhookData['referenceId'] ?? null;

        if (! $referenceId) {
            return response('No reference ID found', 400);
        }

        $transaction = Transaction::where('reference', $referenceId)
            ->where('driver', 'payrexx')
            ->latest()
            ->first();

        if (! $transaction) {
            return response('Transaction not found for reference: '.$referenceId, 404);
        }

        $status = $webhookData['status'] ?? null;

        if (! $status) {
            return response('No status found in transaction data', 400);
        }

        switch ($status) {
            case 'confirmed':
                return $this->handleConfirmed($transaction, $webhookData);

            case 'waiting':
                return $this->handleWaiting($transaction, $webhookData);

            case 'cancelled':
            case 'declined':
            case 'error':
                return $this->handleFailed($transaction, $webhookData, $status);

            default:
                return response('OK');
        }
    }

    /**
     * Handle a confirmed transaction
     *
     * @return \Illuminate\Http\Response
     */
    protected function handleConfirmed(Transaction $transaction, array $webhookData)
    {
        if ($transaction->status === 'complete') {
            return response('Transaction already processed');
        }

        try {
            $cart = Cart::find($transaction->meta['cart_id'] ?? null);

            if (! $cart) {
                Log::error('Cart not found for transaction', [
                    'transaction_id' => $transaction->id,
                ]);

                return response('Cart not found', 404);
            }

            $order = null;

            if ($transaction->order_id) {
                $order = Order::find($transaction->order_id);

                if (! $order) {
                    Log::error('Order not found for transaction', [
                        'transaction_id' => $transaction->id,
                        'order_id' => $transaction->order_id,
                    ]);

                    return response('Order not found', 404);
                }
            } else {
                $order = $cart->createOrder();

                $transaction->update([
                    'order_id' => $order->id,
                ]);
            }

            $driver = new \FirstPoint\LunarPayrexx\PayrexxPaymentDriver;
            $captureResponse = $driver->capture($transaction);

            if (! $captureResponse->success) {
                Log::error('Failed to capture payment', [
                    'transaction_id' => $transaction->id,
                    'message' => $captureResponse->message,
                ]);

                return response('Failed to capture payment: '.$captureResponse->message, 500);
            }

            $cart->update(['completed_at' => Carbon::now()]);
            $cart->delete();

            event(new \FirstPoint\LunarPayrexx\Events\PayrexxOrderCreated($order, $transaction, $webhookData));

            return response('OK');

        } catch (\Exception $e) {
            Log::error('Failed to process Payrexx webhook', [
                'transaction_id' => $transaction->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return response('Error processing webhook: '.$e->getMessage(), 500);
        }
    }

    /**
     * Handle a waiting transaction (payment in progress)
     *
     * @return \Illuminate\Http\Response
     */
    protected function handleWaiting(Transaction $transaction, array $webhookData)
    {
        if (in_array($transaction->status, ['complete', 'waiting'])) {
            return response('Transaction already processed');
        }

        try {
            $cart = Cart::find($transaction->meta['cart_id'] ?? null);

            if (! $cart) {
                Log::error('Cart not found for transaction', [
                    'transaction_id' => $transaction->id,
                ]);

                return response('Cart not found', 404);
            }

            $order = null;

            if ($transaction->order_id) {
                $order = Order::find($transaction->order_id);

                if (! $order) {
                    Log::error('Order not found for transaction', [
                        'transaction_id' => $transaction->id,
                        'order_id' => $transaction->order_id,
                    ]);

                    return response('Order not found', 404);
                }
            } else {
                $order = $cart->createOrder();

                $transaction->update([
                    'order_id' => $order->id,
                ]);
            }

            $transaction->update([
                'status' => 'waiting',
                'meta' => array_merge((array) $transaction->meta, [
                    'waiting_at' => Carbon::now()->toIso8601String(),
                    'payment_method' => $webhookData['psp'] ?? null,
                ]),
            ]);

            $orderStatus = config('lunar-payrexx.order_statuses.pending', 'payment-pending');

            $order->update([
                'status' => $orderStatus,
                'placed_at' => Carbon::now(),
            ]);

            event(new \FirstPoint\LunarPayrexx\Events\PayrexxPaymentWaiting($order, $transaction, $webhookData));

            return response('OK');
        } catch (\Exception $e) {
            Log::error('Failed to process Payrexx waiting webhook', [
                'transaction_id' => $transaction->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return response('Error processing webhook: '.$e->getMessage(), 500);
        }
    }

    /**
     * Handle a failed transaction
     *
     * @param  string  $status  The specific failure status
     * @return \Illuminate\Http\Response
     */
    protected function handleFailed(Transaction $transaction, array $webhookData, string $status)
    {
        $transaction->update([
            'status' => 'failed',
            'success' => false,
            'meta' => array_merge((array) $transaction->meta, ['failure_reason' => $status]),
        ]);

        $orderStatus = config('lunar-payrexx.order_statuses.failed', 'payment-failed');

        if ($transaction->order_id) {
            $order = Order::find($transaction->order_id);
            if ($order) {
                $order->update([
                    'status' => $orderStatus,
                ]);
            }
        }

        event(new \FirstPoint\LunarPayrexx\Events\PayrexxPaymentFailed($transaction, $webhookData, $status));

        return response('OK');
    }
}
