# Lunar Payrexx Payment Driver

This package provides a Payrexx payment driver for the Lunar e-commerce platform.

## Installation

You can install the package via composer:

```bash
composer require firstpoint/lunar-payrexx-driver
```

You can publish the config file with:

```bash
php artisan vendor:publish --tag="lunar-payrexx-config"
```

## Configuration

Add your Payrexx credentials to your `.env` file:

```env
PAYREXX_INSTANCE=your-instance-name
PAYREXX_API_KEY=your-api-key
PAYREXX_TEST_MODE=true
PAYREXX_ENABLE_LANGUAGE_SUPPORT=true
PAYREXX_DEFAULT_LANGUAGE=en
```

### Order Statuses

You can configure the order statuses used for different payment states in the `config/lunar-payrexx.php` file:

```php
'order_statuses' => [
    'pending' => 'awaiting-payment',
    'success' => 'payment-received',
    'failed' => 'payment-failed'
],
```

### Language Support

The package supports automatic language detection and localization of Payrexx payment pages.

#### Prerequisites

Before using language support, you must configure your Payrexx account:

1. **Enable Multilingual Frontend in Payrexx:**
   - Log in to your Payrexx account
   - Navigate to **Settings > General Settings**
   - In the "Languages" section, check "Enable multilingual frontend"
   - Select and activate the languages you want to support
   - Set a default language

2. **Configure Language Settings:**

You can configure language settings in the `config/lunar-payrexx.php` file:

```php
'language' => [
    'enable_language_support' => true,
    'default_language' => 'en',
    'supported_languages' => [
        'en', 'de', 'fr', 'it', 'es', 'pt', 'nl', 'pl', 'cs', 'sk', 'hu', 'ro', 'bg', 'hr', 'sl', 'lv', 'lt', 'et'
    ],
],
```

#### How it Works

1. **Automatic Language Detection:** The package automatically detects the current application locale using `app()->getLocale()`
2. **Cart-Level Language Override:** You can set a specific language for a cart by adding it to the cart meta: `$cart->meta['language'] = 'de'`
3. **URL Modification:** The payment URL is automatically modified to include the language code (e.g., `https://yourinstance.payrexx.com/en/pay?tid=123`)
4. **Fallback:** If the detected language is not supported, it falls back to the configured default language

#### Environment Variables

```env
# Enable/disable language support
PAYREXX_ENABLE_LANGUAGE_SUPPORT=true

# Default language code (must be enabled in your Payrexx account)
PAYREXX_DEFAULT_LANGUAGE=en
```

#### Setting Language Per Cart

You can override the language for specific carts:

```php
// Set German as the language for this cart
$cart->update([
    'meta' => [
        ...$cart->meta,
        'language' => 'de'
    ]
]);
```

#### Supported Language Codes

The package supports the following language codes (ensure they are enabled in your Payrexx account):
- `en` - English
- `de` - German
- `fr` - French
- `it` - Italian
- `es` - Spanish
- `pt` - Portuguese
- `nl` - Dutch
- `pl` - Polish
- `cs` - Czech
- `sk` - Slovak
- `hu` - Hungarian
- `ro` - Romanian
- `bg` - Bulgarian
- `hr` - Croatian
- `sl` - Slovenian
- `lv` - Latvian
- `lt` - Lithuanian
- `et` - Estonian

### Custom Redirect Routes

You can customize the redirect routes used after payment completion using the `LunarPayrexx` facade in your `AppServiceProvider`. This approach preserves Laravel's config caching capabilities while allowing dynamic route generation.

Add the following to your `AppServiceProvider::boot()` method:

```php
use FirstPoint\LunarPayrexx\Facades\LunarPayrexx;

public function boot()
{
    // Set custom redirect handlers
    LunarPayrexx::setSuccessRedirect(function ($reference) {
        return route('payment.success', ['ref' => $reference]);
    });

    LunarPayrexx::setFailedRedirect(function () {
        return route('payment.failed');
    });

    LunarPayrexx::setCancelRedirect(function () {
        return route('payment.cancel');
    });
}
```

#### Examples for Locale-Prefixed Routes

If your routes are prefixed with locale (e.g., `/en/`, `/fr/`), you can use the current locale:

```php
use FirstPoint\LunarPayrexx\Facades\LunarPayrexx;

public function boot()
{
    LunarPayrexx::setSuccessRedirect(function ($reference) {
        return route('payment.success', [
            'locale' => app()->getLocale(),
            'ref' => $reference
        ]);
    });

    LunarPayrexx::setFailedRedirect(function () {
        return route('payment.failed', ['locale' => app()->getLocale()]);
    });

    LunarPayrexx::setCancelRedirect(function () {
        return route('payment.cancel', ['locale' => app()->getLocale()]);
    });
}
```

#### Custom Route Controllers

When using custom routes, make sure your controllers handle the parameters correctly:

- **Success route**: Should accept a `ref` parameter containing the payment reference
- **Failed/Cancel routes**: Can optionally accept error information from query parameters

Example custom controller:
```php
class CustomPaymentController extends Controller
{
    public function success(Request $request)
    {
        $reference = $request->get('ref');
        // Handle success logic
        return view('payment.success', compact('reference'));
    }

    public function failed(Request $request)
    {
        // Handle failed payment
        return view('payment.failed');
    }
}
```

## Usage

### Basic Usage

```php
use Lunar\Facades\Payments;

// Redirect the customer to the Payrexx payment page
return Payments::driver('payrexx')->cart($cart)->redirectToPayment();
```

## Payment Flow

1. When a payment is initiated, a draft order is created with a "pending" status (configured in `lunar-payrexx.order_statuses.pending`)
2. The customer is redirected to the Payrexx payment page
3. After payment, the customer is redirected back to your site
4. The payment status is shown to the customer (with a message indicating the payment is being processed)
5. Payrexx sends a webhook notification with the payment result
6. The webhook handler processes the payment based on its status:
   - If "waiting": The order is undrafted (placed) but remains in a pending state (configured in `lunar-payrexx.order_statuses.pending`)
   - If "confirmed": The order is marked as placed and its status is updated (configured in `lunar-payrexx.order_statuses.success`)
   - If "cancelled/declined/error": The order status is updated to reflect the failure (configured in `lunar-payrexx.order_statuses.failed`)
7. If the payment is successful, the cart is marked as completed
8. An event is dispatched based on the payment result (`PayrexxPaymentWaiting`, `PayrexxOrderCreated`, or `PayrexxPaymentFailed`)

**Important**: The actual payment verification happens in the webhook handler. The redirect pages are only for user feedback.

### Handling Cart Changes Between Payment Attempts

If a customer exits the payment portal, modifies their cart, and then initiates payment again:

1. The package will detect the existing draft order
2. It will update the order to reflect the current cart contents
3. This ensures that the order always matches what the customer intends to purchase
4. No duplicate orders are created in this process

## Customizing Views

You can publish the views to customize them:

```bash
php artisan vendor:publish --tag="lunar-payrexx-views"
```

This will publish the views to `resources/views/vendor/lunar-payrexx/`.

## Webhook Configuration

In your Payrexx dashboard, set up a webhook to point to:

```
POST /payrexx/webhook
```

Make sure to configure this webhook URL in your Payrexx dashboard:
```
https://your-site.com/payrexx/webhook
```

The webhook handler processes the following Payrexx transaction statuses:

| Payrexx Status | Transaction Status | Order Status | Description |
|----------------|-------------------|--------------|-------------|
| waiting | waiting | pending* | Payment is in progress (e.g., invoice payment) |
| confirmed | complete | success* | Payment was successful |
| cancelled | failed | failed* | Payment was cancelled |
| declined | failed | failed* | Payment was declined |
| error | failed | failed* | Technical error occurred |

*Order statuses are configured in the `config/lunar-payrexx.php` file.

## Refund

Payrexx refunds must be processed manually in the Payrexx dashboard. When you click the refund button in the Lunar admin, it will only create a refund transaction record in your database to track the refund request

**Important**: After clicking the refund button in Lunar, you must log in to your Payrexx dashboard to actually process the refund.

## Events

This package dispatches the following events that you can listen for in your application:

### PayrexxPaymentWaiting

This event is dispatched when a payment is in progress (typically for invoice payments or other methods with delayed confirmation):

```php
use FirstPoint\LunarPayrexx\Events\PayrexxPaymentWaiting;

// In your EventServiceProvider
protected $listen = [
    PayrexxPaymentWaiting::class => [
        SendPaymentInstructionsEmail::class,
    ],
];
```

The event contains the following properties:

- `$order`: The Lunar Order model
- `$transaction`: The Lunar Transaction model
- `$webhookData`: The raw webhook data from Payrexx

### PayrexxOrderCreated

This event is dispatched when an order is created after a successful payment:

```php
use FirstPoint\LunarPayrexx\Events\PayrexxOrderCreated;

// In your EventServiceProvider
protected $listen = [
    PayrexxOrderCreated::class => [
        SendOrderConfirmationEmail::class,
    ],
];
```

The event contains the following properties:

- `$order`: The Lunar Order model
- `$transaction`: The Lunar Transaction model
- `$webhookData`: The raw webhook data from Payrexx

### PayrexxPaymentFailed

This event is dispatched when a payment fails:

```php
use FirstPoint\LunarPayrexx\Events\PayrexxPaymentFailed;

// In your EventServiceProvider
protected $listen = [
    PayrexxPaymentFailed::class => [
        NotifyCustomerAboutFailedPayment::class,
    ],
];
```

The event contains the following properties:

- `$transaction`: The Lunar Transaction model
- `$webhookData`: The raw webhook data from Payrexx
- `$status`: The specific failure status from Payrexx (e.g., 'cancelled', 'declined', 'error')

## Security Considerations

This package follows Payrexx's recommended security practices:

1. The payment status is only validated and processed in the webhook handler
2. The redirect controllers only show status pages to the user
3. Orders are created with a pending status until payment is confirmed
4. Transaction IDs are validated to prevent duplicate processing
