<?php

namespace FirstPoint\LunarPayrexx;

use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Session;
use Lunar\Base\DataTransferObjects\PaymentAuthorize;
use Lunar\Base\DataTransferObjects\PaymentCapture;
use Lunar\Base\DataTransferObjects\PaymentRefund;
use Lunar\Models\Contracts\Cart;
use Lunar\Models\Contracts\Transaction;
use Lunar\Models\Transaction as TransactionModel;
use Lunar\PaymentTypes\AbstractPayment;
use Payrexx\Models\Request\Gateway;
use Payrexx\Payrexx;

/**
 * Payrexx Payment Driver for Lunar
 *
 * This driver provides a simple way to integrate Payrexx payments with Lunar.
 * It creates a payment URL that customers can be redirected to, and handles
 * the payment status through webhooks.
 */
class PayrexxPaymentDriver extends AbstractPayment
{
    protected ?Payrexx $payrexx = null;

    protected ?Cart $cart = null;

    /**
     * Set up the Payrexx client with configuration
     */
    protected function setupPayrexx(): void
    {
        if (! $this->payrexx) {
            $this->payrexx = new Payrexx(
                config('lunar-payrexx.instance'),
                config('lunar-payrexx.api_key')
            );
        }
    }

    /**
     * Create a Payrexx gateway and transaction
     *
     * @return object|null The Payrexx gateway response or null on failure
     *
     * @throws \Exception
     */
    protected function createPayrexxGateway()
    {
        if (! $this->cart) {
            throw new \Exception('No cart provided');
        }

        $this->setupPayrexx();

        $gateway = new Gateway;
        $gateway->setAmount($this->cart->total->value);
        $gateway->setCurrency($this->cart->currency->code);
        $gateway->setPurpose('Order from '.config('app.name'));
        $gateway->addField('forename', $this->cart->billingAddress?->first_name);
        $gateway->addField('surname', $this->cart->billingAddress?->last_name);
        $gateway->addField('email', $this->cart->billingAddress?->contact_email);
        $gateway->addField('phone', $this->cart->billingAddress?->contact_phone);
        $gateway->addField('address', $this->cart->billingAddress?->address_1);
        $gateway->addField('city', $this->cart->billingAddress?->city);
        $gateway->addField('postcode', $this->cart->billingAddress?->postcode);
        $gateway->addField('country', $this->cart->billingAddress?->country?->iso2);

        if (!empty($this->cart->meta['paymentMethod'])) {
            $gateway->setPm([$this->cart->meta['paymentMethod']]);
        }

        $uniqueReference = 'px_'.uniqid().'_'.$this->cart->id;

        $gateway->setReferenceId($uniqueReference);

        $gateway->setSuccessRedirectUrl($this->getSuccessRedirectUrl($uniqueReference));
        $gateway->setFailedRedirectUrl($this->getFailedRedirectUrl());
        $gateway->setCancelRedirectUrl($this->getCancelRedirectUrl());
        $gateway->setSkipResultPage(true);

        try {
            $response = $this->payrexx->create($gateway);

            $meta = [
                'cart_id' => $this->cart->id,
                'gateway_id' => $response->getId(),
                'payment_url' => $response->getLink(),
            ];

            $order = $this->cart->order;

            $order = $this->cart->createOrder(
                allowMultipleOrders: false,
                orderIdToUpdate: ($order && $order->placed_at === null) ? $order->id : null
            );

            $order->update([
                'status' => config('lunar-payrexx.order_statuses.pending', 'payment-pending'),
                'meta' => [
                    ...$order->meta,
                    'locale' => app()->getLocale(),
                ],
            ]);

            TransactionModel::create([
                'order_id' => $order->id,
                'type' => 'intent',
                'driver' => 'payrexx',
                'amount' => $this->cart->total->value,
                'reference' => $uniqueReference,
                'status' => 'pending',
                'success' => true,
                'card_type' => 'payrexx',
                'meta' => $meta,
            ]);

            return $response;
        } catch (\Exception $e) {
            Log::error('Failed to create Payrexx gateway', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            throw $e;
        }
    }

    /**
     * Authorize a payment for the current cart
     *
     * Note: Payrexx doesn't support direct authorization. This method creates a payment
     * page and returns the URL for redirection. The actual authorization happens on
     * Payrexx's hosted payment page.
     *
     * This method is primarily for internal use. Integrators should use redirectToPayment() instead.
     */
    public function authorize(?PaymentAuthorize $payment = null): PaymentAuthorize
    {
        if (! $payment) {
            $payment = new PaymentAuthorize(
                success: false,
                message: 'Initializing Payrexx payment',
                orderId: null,
                paymentType: 'payrexx'
            );
        }

        try {
            $paymentUrl = $this->getPaymentUrl();

            if (! $paymentUrl) {
                $payment->success = false;
                $payment->message = 'Failed to create Payrexx payment page';

                return $payment;
            }

            $payment->success = true;
            $payment->message = 'Redirect to Payrexx payment page';

            return $payment;
        } catch (\Exception $e) {
            $payment->success = false;
            $payment->message = 'Failed to create Payrexx payment: '.$e->getMessage();

            return $payment;
        }
    }

    /**
     * Get the payment URL for the current cart
     *
     * This is the main method to use with Payrexx. It creates a payment page
     * and returns the URL that the customer should be redirected to.
     *
     *
     * @return string|null The URL to redirect the customer to, or null on failure
     */
    public function getPaymentUrl(): ?string
    {
        $paymentUrl = $this->createPayrexxGateway()->getLink();

        if ($paymentUrl && config('lunar-payrexx.language.enable_language_support', true)) {
            $paymentUrl = $this->addLanguageToPaymentUrl($paymentUrl);
        }

        return $paymentUrl;
    }

    /**
     * Add language code to the payment URL
     *
     * Modifies the Payrexx payment URL to include the appropriate language code
     * based on the current application locale or cart configuration.
     *
     * @param string $paymentUrl The original payment URL from Payrexx
     * @return string The modified URL with language code
     */
    protected function addLanguageToPaymentUrl(string $paymentUrl): string
    {
        // Get the current locale
        $locale = $this->cart->meta['language'] ?? app()->getLocale();

        // Get the supported languages from config
        $supportedLanguages = config('lunar-payrexx.language.supported_languages', ['en']);
        $defaultLanguage = config('lunar-payrexx.language.default_language', 'en');

        // Normalize locale (convert 'en_US' to 'en', etc.)
        $languageCode = substr($locale, 0, 2);

        // Use default language if the current language is not supported
        if (!in_array($languageCode, $supportedLanguages)) {
            $languageCode = $defaultLanguage;
        }

        // Parse the URL and inject the language code
        $parsedUrl = parse_url($paymentUrl);

        // Build the new path with language code
        $path = $parsedUrl['path'] ?? '';

        // Check if language code is already in the URL
        if (!preg_match('/^\/[a-z]{2}\//', $path)) {
            // Add language code to the beginning of the path
            $path = '/' . $languageCode . $path;
        }

        // Rebuild the URL
        $scheme = $parsedUrl['scheme'] ?? 'https';
        $host = $parsedUrl['host'] ?? '';
        $query = isset($parsedUrl['query']) ? '?' . $parsedUrl['query'] : '';
        $fragment = isset($parsedUrl['fragment']) ? '#' . $parsedUrl['fragment'] : '';

        return $scheme . '://' . $host . $path . $query . $fragment;
    }

    /**
     * Get the redirect URL for the payment
     */
    protected function getRedirectUrl(): ?string
    {
        $transaction = TransactionModel::where('driver', 'payrexx')
            ->where('type', 'intent')
            ->whereJsonContains('meta->cart_id', $this->cart->id)
            ->latest()
            ->first();

        return $transaction->meta['payment_url'] ?? null;
    }

    /**
     * Redirect to the Payrexx payment page
     *
     * This is the primary method that integrators should use to process payments with Payrexx.
     * It handles creating the payment intent and redirecting to the Payrexx payment page.
     *
     * @return \Illuminate\Http\RedirectResponse
     *
     * @throws \Exception
     */
    public function redirectToPayment()
    {
        $payment = $this->authorize();

        if (! $payment->success) {
            throw new \Exception('Failed to create payment: '.$payment->message);
        }

        $url = $this->getRedirectUrl();

        if (! $url) {
            throw new \Exception('Failed to get payment URL');
        }

        return redirect($url);
    }

    /**
     * Capture a payment
     *
     * Note: Payrexx payments are automatically captured.
     * This method is provided for compatibility with the Lunar API.
     *
     * @param  int  $amount
     */
    public function capture(Transaction $transaction, $amount = 0): PaymentCapture
    {
        if (! $amount) {
            $amount = $transaction->amount;
        }

        TransactionModel::create([
            'parent_transaction_id' => $transaction->id,
            'order_id' => $transaction->order_id,
            'type' => 'capture',
            'driver' => 'payrexx',
            'amount' => $amount,
            'reference' => $transaction->reference,
            'status' => 'complete',
            'success' => true,
            'card_type' => 'payrexx',
            'meta' => (array) $transaction->meta,
        ]);

        TransactionModel::where('id', $transaction->id)->update([
            'status' => 'complete',
            'captured_at' => Carbon::now(),
        ]);

        if ($transaction->order) {
            $successStatus = config('lunar-payrexx.order_statuses.success', 'paid');
            $transaction->order->update([
                'status' => $successStatus,
                'placed_at' => Carbon::now(),
            ]);
        }

        return new PaymentCapture(
            success: true,
            message: 'Payment captured successfully'
        );
    }

    /**
     * Refund a payment
     *
     * Note: Payrexx refunds must be handled manually in the Payrexx dashboard.
     * This method is provided for compatibility with the Lunar API.
     *
     * @param  string|null  $notes
     */
    public function refund(Transaction $transaction, int $amount = 0, $notes = null): PaymentRefund
    {
        TransactionModel::create([
            'parent_transaction_id' => $transaction->id,
            'order_id' => $transaction->order_id,
            'type' => 'refund',
            'driver' => 'payrexx',
            'amount' => $amount ?: $transaction->amount,
            'reference' => $transaction->reference,
            'status' => 'complete',
            'success' => true,
            'card_type' => 'payrexx',
            'meta' => array_merge((array) $transaction->meta, [
                'notes' => $notes,
                'manual_refund_required' => true,
                'refund_requested_at' => Carbon::now()->toIso8601String(),
            ]),
        ]);

        return new PaymentRefund(
            success: true,
            message: 'IMPORTANT: Payrexx refunds must be processed manually in the Payrexx dashboard. This refund has been recorded in Lunar but has NOT been processed in Payrexx.'
        );
    }

    /**
     * Get the success redirect URL
     */
    protected function getSuccessRedirectUrl(string $reference): string
    {
        return app('lunar-payrexx-redirect-manager')->getSuccessRedirectUrl($reference);
    }

        /**
     * Get the failed redirect URL
     */
    protected function getFailedRedirectUrl(): string
    {
        return app('lunar-payrexx-redirect-manager')->getFailedRedirectUrl();
    }

    /**
     * Get the cancel redirect URL
     */
    protected function getCancelRedirectUrl(): string
    {
        return app('lunar-payrexx-redirect-manager')->getCancelRedirectUrl();
    }
}
