<?php

namespace FirstPoint\LunarHeylight\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Routing\Controller;
use Lunar\Models\Order;
use Lunar\Models\Cart;

class WebhookController extends Controller
{
    public function handlePaymentCompleted(Request $request)
    {
        if (!$this->verifyWebhookSignature($request)) {
            return response()->json(['error' => 'Invalid signature'], 401);
        }

        $payload = $request->all();
        $cart = Cart::find($payload['token']);
        
        if (!$cart) {
            return response()->json(['error' => 'Cart not found'], 404);
        }

        $order = Order::whereCartId($cart->id)->first();
        if (!$order) {
            return response()->json(['error' => 'Order not found'], 404);
        }

        $order->update(['status' => $this->mapPaymentStatus($payload['status'])]);
        return response()->json(['success' => true]);
    }

    protected function mapPaymentStatus(string $status): string
    {
        return match($status) {
            'success' => 'payment-received',
            'cancelled' => 'payment-failed',
            'awaiting_confirmation', 'pending' => 'pending',
            default => 'payment-failed'
        };
    }

    protected function verifyWebhookSignature(Request $request): bool
    {
        $signature = $request->header('X-Heylight-Signature');
        $payload = $request->getContent();
        $secret = config('lunar.payments.heylight.webhook_secret');

        return hash_equals(
            $signature, 
            hash_hmac('sha256', $payload, $secret)
        );
    }

    public function handlePaymentFailed(Request $request)
    {
        if (!$this->verifyWebhookSignature($request)) {
            return response()->json(['error' => 'Invalid signature'], 401);
        }

        $payload = $request->all();
        
        $cart = Cart::find($payload['token']);
        if ($cart) {
            $order = Order::whereCartId($cart->id)->first();
            if ($order) {
                $order->update(['status' => 'payment-failed']);
            }
        }

        return response()->json(['success' => true]);
    }
}
