<?php

namespace FirstPoint\LunarHeylight;

use GuzzleHttp\Client;
use Lunar\Base\DataTransferObjects\PaymentAuthorize;
use Lunar\Base\DataTransferObjects\PaymentCapture;
use Lunar\Base\DataTransferObjects\PaymentRefund;
use Lunar\Models\Cart;
use Lunar\Models\Transaction;
use Lunar\PaymentTypes\AbstractPayment;

class HeylightPaymentDriver extends AbstractPayment
{
    protected ?Cart $cart = null;
    protected Client $client;
    protected array $config;
    protected ?string $accessToken = null;

    public function __construct()
    {
        $this->config = config('lunar-heylight');
        $this->initializeClient();
        $this->authenticate();
    }

    protected function initializeClient(): void
    {
        $baseUrl = $this->config['sandbox'] ?? true
            ? 'https://sbx-origination.heidipay.io'
            : 'https://origination.heidipay.com';

        $this->client = new Client([
            'base_uri' => $baseUrl,
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ]
        ]);
    }

    public function cart(Cart $cart): self
    {
        $this->cart = $cart;
        return $this;
    }

    public function authorize(): PaymentAuthorize
    {
        try {
            $response = $this->createPayment();
            
            if ($response->success()) {
                return new PaymentAuthorize(
                    success: true,
                    message: $response->message(),
                    orderId: $this->cart?->id
                );
            }

            return new PaymentAuthorize(
                success: false,
                message: $response->message(),
				orderId: $this->cart?->id
            );
        } catch (\Exception $e) {
            return new PaymentAuthorize(
                success: false,
                message: $e->getMessage(),
				orderId: $this->cart?->id
            );
        }
    }

    public function getPaymentUrl(): ?string
    {
        try {
            $response = $this->createPayment();
            return $response->getPaymentUrl();
        } catch (\Exception $e) {
            report($e);
            return null;
        }
    }

    protected function createPayment(): HeylightPaymentResponse
    {
        if (!$this->cart) {
            throw new \Exception('Cart is required for payment creation');
        }

        $cart = $this->cart->calculate();
        $billingAddress = $cart->billingAddress;
        
        if (!$billingAddress) {
            throw new \Exception('Billing address is required for payment creation');
        }

        $payload = [
            'amount' => [
                'currency' => $cart->currency->code,
                'amount' => number_format($cart->total->value, 2, '.', '')
            ],
            'amount_format' => 'DECIMAL',
            'customer_details' => [
                'email_address' => $billingAddress->contact_email,
                'first_name' => $billingAddress->first_name,
                'last_name' => $billingAddress->last_name,
                'contact_number' => $billingAddress->contact_phone ?? ''
            ],
            'redirect_urls' => [
                'success_url' => $this->config['payment']['success_url'] ?? route('heylight.webhook.payment-completed'),
                'failure_url' => $this->config['payment']['failure_url'] ?? route('heylight.webhook.payment-failed')
            ],
            'webhooks' => [
                'mapping_scheme' => 'DEFAULT',
                'status_url' => route('heylight.webhook.payment-completed'),
                'token' => $cart->id
            ],
            'store_id' => $this->config['payment']['store_id'] ?? 'ecommerce',
            'allowed_terms' => $this->config['payment']['allowed_terms'] ?? [3, 6, 12],
            'order_reference' => $cart->id,
            'language' => app()->getLocale()
        ];

        $response = $this->client->post('api/checkout/v1/init/', [
            'json' => $payload
        ]);

        return new HeylightPaymentResponse($response);
    }

    public function refund(Transaction $transaction, int $amount = 0, $notes = null): PaymentRefund
    {
        // Heylight doesn't support refunds through API yet
        return new PaymentRefund(
            success: false,
            message: 'Refunds are not supported through the API'
        );
    }

    public function capture(Transaction $transaction, $amount = 0): PaymentCapture
    {
        // Heylight automatically captures payments, no manual capture needed
        return new PaymentCapture(
            success: true,
            message: 'Payments are automatically captured'
        );
    }

    protected function authenticate(): void
    {
        try {
            $response = $this->client->post('auth/v1/generate/', [
                'json' => [
                    'merchant_key' => $this->config['merchant_key'],
                ],
                'headers' => [
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ]
            ]);

            $data = json_decode($response->getBody()->getContents(), true);
            
            if (!isset($data['status']) || $data['status'] !== 'success' || empty($data['data']['token'])) {
                throw new \Exception('Authentication failed: Invalid response format');
            }
            
            $this->accessToken = $data['data']['token'];
            
            // Update client with authentication token
            $this->client = new Client([
                'base_uri' => $this->client->getConfig('base_uri'),
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->accessToken,
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json',
                ]
            ]);
        } catch (\Exception $e) {
            throw new \Exception('Failed to authenticate with Heylight: ' . $e->getMessage());
        }
    }
}
