# Laravel Pull

A Laravel package that simplifies copying production data (database and storage files) to your local development environment. Perfect for developers who need to work with real production data locally.

## Features

- 🗄️ **Database Pulling**: Stream production database data directly to your local database using mysqldump (over SSH or locally)
- 📁 **Storage Pulling**: Sync production storage files to local using rsync (over SSH or locally)
- 🏠 **Localhost Support**: Connect directly to databases and storage on the same server (perfect for staging environments)
- 🔒 **SSH Support**: Connect to remote servers via SSH for traditional remote deployments

## Use Cases

### Localhost Connection
Perfect for scenarios where your development and production/staging environments are on the same server:
- **Staging to Development**: Copy staging data to your development environment on the same server
- **Production to Staging**: Refresh staging environment with production data
- **Multiple Laravel installations**: Copy data between different Laravel projects on the same server
- **Shared hosting**: When you have multiple environments on a shared hosting provider

### SSH Connection
Ideal for traditional remote server setups:
- **Remote Production**: Access production data from a completely separate server
- **Cloud Deployments**: Pull data from cloud-hosted production environments
- **Distributed Teams**: Access shared production/staging servers
- **Secure Remote Access**: When you need encrypted connections to sensitive data

## Installation

Add the following to your `composer.json` file:
```json
{
    "repositories": [
        {
            "type": "composer",
            "url": "https://package.firstpoint.ch"
        },
    ],
}
```

Then you can install the package via composer:

```bash
composer require firstpoint-ch/laravel-pull --dev
```

Publish the config file:

```bash
php artisan vendor:publish --provider="FirstpointCh\LaravelPull\LaravelPullServiceProvider"
```

## Configuration

### Connection Types

The package supports two connection types:

#### SSH Connection (Default)
Connect to a remote server via SSH. This is the traditional way and perfect for accessing production servers.

```env
# Use SSH connection (default)
PROD_CONNECTION_TYPE=ssh
PROD_SSH_USERNAME=your-ssh-username
PROD_SSH_HOST=your-production-server.com
PROD_SSH_PORT=22
```

#### Localhost Connection
Connect directly to databases and storage on the same server. Perfect for staging environments or when your development and production environments are on the same server.

```env
# Use localhost connection
PROD_CONNECTION_TYPE=localhost
```

When using localhost connection:
- No SSH credentials are required
- Database connections are made directly to the specified host/port
- Storage paths should use absolute paths on the filesystem
- Rsync operations are performed locally without SSH

### Storage Paths

Configure multiple storage paths to sync different directories from production:

```php
// config/pull.php
'storage' => [
    'paths' => [
        [
            'name' => 'public',
            'remote_path' => '~/sites/example.com/storage/app/public/',
            'local_path' => 'storage/app/public/',
        ],
    ],
],
```

### Environment Variables

Add the following environment variables to your `.env` file:

#### For SSH Connections
```env
# Connection type (ssh is default)
PROD_CONNECTION_TYPE=ssh

# Production SSH Connection
PROD_SSH_USERNAME=your-ssh-username
PROD_SSH_HOST=your-production-server.com
PROD_SSH_PORT=22

# Production Database
PROD_DB_HOST=127.0.0.1
PROD_DB_PORT=3306
PROD_DB_DATABASE=production_database
PROD_DB_USERNAME=prod_db_user
PROD_DB_PASSWORD=prod_db_password
```

#### For Localhost Connections
```env
# Connection type
PROD_CONNECTION_TYPE=localhost

# Production Database (direct connection)
PROD_DB_HOST=127.0.0.1  # or your database host
PROD_DB_PORT=3306
PROD_DB_DATABASE=production_database
PROD_DB_USERNAME=prod_db_user
PROD_DB_PASSWORD=prod_db_password

# Storage paths (use absolute paths)
PROD_STORAGE_PUBLIC_PATH=/var/www/production/storage/app/public/
```

### SQLite Database Support

For SQLite databases, the package will copy the database file directly instead of using mysqldump:
- **SSH connections**: Uses SCP to copy the file
- **Localhost connections**: Uses local file copy (`cp` command)

#### SQLite with SSH
```env
# Production Database (SQLite over SSH)
PROD_CONNECTION_TYPE=ssh
PROD_DB_TYPE=sqlite
PROD_SQLITE_PATH=~/sites/yoursite.com/database/database.sqlite
```

#### SQLite with Localhost
```env
# Production Database (SQLite locally)
PROD_CONNECTION_TYPE=localhost
PROD_DB_TYPE=sqlite
PROD_SQLITE_PATH=/var/www/production/database/database.sqlite
```

Make sure your local database is also configured to use SQLite in your `config/database.php` and the pull configuration:

```php
// config/pull.php
'database' => [
    'type' => env('PROD_DB_TYPE', 'mysql'), // Set to 'sqlite' for SQLite

    // SQLite settings
    'sqlite' => [
        'remote_path' => env('PROD_SQLITE_PATH'),
    ],

    // ... other settings
],

'local_database' => [
    'connection' => env('DB_CONNECTION', 'mysql'), // Should be 'sqlite' for SQLite
    'database' => env('DB_DATABASE'),
    // ... other settings
],
```

**Note:** When using SQLite, both your production and local databases must be SQLite. The package will validate this configuration and throw an error if there's a mismatch.

### Database Types Supported

- **MySQL** (default): Uses `mysqldump` and `mysql` commands
- **SQLite**: Direct file copy using SCP

## Usage

### Artisan Commands

Pull everything (database + storage):
```bash
php artisan pull
```

Pull only database:
```bash
php artisan pull:database

# Skip confirmation prompt
php artisan pull:database --force
```

Pull only storage:
```bash
php artisan pull:storage

# Pull specific storage path
php artisan pull:storage public
```

## How It Works

### Database Pulling

#### SSH Mode
1. Connects to your production server via SSH
2. Runs `mysqldump` with optimized settings on the production database
3. Streams the dump directly to your local MySQL database via SSH tunnel
4. Uses `REPLACE INTO` statements to handle duplicate keys gracefully

#### Localhost Mode
1. Runs `mysqldump` directly on the local server
2. Streams the dump directly to your local MySQL database
3. Perfect for staging environments on the same server

### Storage Pulling

#### SSH Mode
1. Uses `rsync` over SSH to sync files from remote server
2. Automatically detects rsync version for optimal progress reporting
3. Preserves file permissions and timestamps
4. Removes local files that no longer exist in production (with `--delete`)

#### Localhost Mode
1. Uses `rsync` locally to sync files between directories
2. No SSH overhead - direct filesystem operations
3. Preserves file permissions and timestamps
4. Perfect for copying from staging to development on the same server

## Requirements

- PHP 8.3+
- Laravel 11.0+
- `mysqldump` and `mysql` commands available (for MySQL databases)
- `rsync` command available
- **For SSH connections**: SSH access to production server
- **For localhost connections**: Direct access to production database and storage paths

## Security Considerations

- Store production credentials securely in environment variables
- Use SSH key authentication when possible
- Consider using read-only database users for pulling
- Be cautious when pulling to shared development environments
- The package includes confirmation prompts for destructive operations

## Testing

```bash
composer test
```
