<?php

namespace FirstpointCh\LaravelPull;

use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Process;
use InvalidArgumentException;
use RuntimeException;

class LaravelPull
{
    protected array $config;

    protected ?\Closure $progressCallback = null;

    public function __construct()
    {
        $this->config = config('pull');
    }

    public function setProgressCallback(?\Closure $callback): self
    {
        $this->progressCallback = $callback;

        return $this;
    }

    protected function progress(string $message): void
    {
        if ($this->progressCallback) {
            ($this->progressCallback)($message);
        }
    }

    public function pullDatabase(): bool
    {
        $this->progress('Validating database configuration...');
        $this->validateDatabaseConfig();

        $databaseType = $this->getDatabaseType();
        $this->progress("Detected database type: {$databaseType}");

        if ($databaseType === 'sqlite') {
            return $this->pullSqliteDatabase();
        }

        return $this->pullMysqlDatabase();
    }

    protected function pullSqliteDatabase(): bool
    {
        try {
            $this->progress('Preparing SQLite database file copy...');
            $remotePath = $this->config['database']['sqlite']['remote_path'];
            $localPath = database_path(basename($remotePath));

            $this->progress("Copying SQLite database from {$remotePath} to {$localPath}");
            $scpCommand = $this->buildScpCommand($remotePath, $localPath);

            $this->progress('Executing SCP transfer...');
            $result = Process::run($scpCommand);

            if ($result->failed()) {
                throw new RuntimeException("SQLite database copy failed: {$result->errorOutput()}");
            }

            $this->progress('SQLite database copy completed successfully');

            return true;

        } catch (\Exception $e) {
            throw $e;
        }
    }

    protected function pullMysqlDatabase(): bool
    {
        try {
            $this->progress('Building database dump command...');
            $remoteDumpCommand = $this->buildRemoteDumpCommand();

            $this->progress('Building local import command...');
            $localImportCommand = $this->buildLocalImportCommand();

            $this->progress('Establishing SSH connection and executing database dump...');
            $sshCommand = $this->buildSshCommand($remoteDumpCommand);
            $fullCommand = "{$sshCommand} | {$localImportCommand}";

            $this->progress('Transferring and importing database (this may take a while)...');
            $result = Process::run($fullCommand);

            if ($result->failed()) {
                throw new RuntimeException("Database pull failed: {$result->errorOutput()}");
            }

            $this->progress('Database import completed successfully');

            return true;

        } catch (\Exception $e) {
            throw $e;
        }
    }

    public function pullStorage(?string $pathName = null): bool
    {
        $this->progress('Validating storage configuration...');
        $this->validateStorageConfig();

        $this->progress('Loading storage paths configuration...');
        $storagePaths = $this->getStoragePaths();

        if ($pathName) {
            $this->progress("Filtering for storage path: {$pathName}");
            $storagePaths = array_filter($storagePaths, fn ($path) => $path['name'] === $pathName);
            if (empty($storagePaths)) {
                throw new InvalidArgumentException("Storage path '{$pathName}' not found in configuration");
            }
        }

        $totalPaths = count($storagePaths);
        $this->progress("Found {$totalPaths} storage path(s) to sync");

        $allSuccessful = true;
        $errors = [];
        $currentPath = 0;

        foreach ($storagePaths as $path) {
            $currentPath++;
            $this->progress("Syncing path {$currentPath}/{$totalPaths}: {$path['name']}");

            try {
                $this->pullSingleStoragePath($path);
                $this->progress("✓ Successfully synced: {$path['name']}");
            } catch (\Exception $e) {
                $allSuccessful = false;
                $errors[] = "Failed to pull '{$path['name']}': {$e->getMessage()}";
                $this->progress("✗ Failed to sync: {$path['name']} - {$e->getMessage()}");
            }
        }

        if (! $allSuccessful) {
            throw new RuntimeException('Some storage pulls failed: '.implode(', ', $errors));
        }

        $this->progress('All storage paths synced successfully');

        return true;
    }

    protected function pullSingleStoragePath(array $path): void
    {
        $localPath = base_path($path['local_path']);

        // Ensure local directory exists
        if (! File::exists($localPath)) {
            $this->progress("Creating local directory: {$path['local_path']}");
            File::makeDirectory($localPath, 0755, true);
        }

        $this->progress("Building rsync command for: {$path['remote_path']} → {$path['local_path']}");
        $rsyncCommand = $this->buildRsyncCommand($path['remote_path'], $path['local_path']);

        $this->progress('Executing rsync (files will be transferred with progress indicators)...');
        $result = Process::run($rsyncCommand);

        if ($result->failed()) {
            throw new RuntimeException("Storage pull failed: {$result->errorOutput()}");
        }
    }

    public function pullAll(): bool
    {
        $this->progress('Starting complete data pull (database + storage)...');

        $this->progress('Step 1/2: Pulling database...');
        $databaseSuccess = $this->pullDatabase();

        $this->progress('Step 2/2: Pulling storage files...');
        $storageSuccess = $this->pullStorage();

        $this->progress('Complete data pull finished');

        return $databaseSuccess && $storageSuccess;
    }

    protected function buildRemoteDumpCommand(): string
    {
        $dbConfig = $this->config['database'];

        $command = [
            'mysqldump',
            "--user={$dbConfig['username']}",
            "--password={$dbConfig['password']}",
            "--host={$dbConfig['host']}",
            "--port={$dbConfig['port']}",
        ];

        // Add dump options
        $command = array_merge($command, $dbConfig['dump_options']);

        // Add database name
        $command[] = $dbConfig['database'];

        return implode(' ', array_map('escapeshellarg', $command));
    }

    protected function buildLocalImportCommand(): string
    {
        $localDb = $this->config['local_database'];

        $command = [
            'mysql',
            "--user={$localDb['username']}",
            '--force',
        ];

        if (! empty($localDb['password'])) {
            $command[] = "--password={$localDb['password']}";
        }

        $command[] = $localDb['database'];

        return implode(' ', array_map('escapeshellarg', $command));
    }

    protected function buildSshCommand(string $remoteCommand): string
    {
        $sshConfig = $this->config['ssh'];

        $sshOptions = [];
        foreach ($sshConfig['options'] as $key => $value) {
            $sshOptions[] = "-o {$key}={$value}";
        }

        $sshOptionsString = implode(' ', $sshOptions);
        $sshHost = "{$sshConfig['username']}@{$sshConfig['host']}";

        return "ssh {$sshOptionsString} {$sshHost} {$remoteCommand}";
    }

    protected function buildRsyncCommand(string $remotePath, string $localPath): string
    {
        $storageConfig = $this->config['storage'];
        $sshConfig = $this->config['ssh'];

        $rsyncOptions = implode(' ', $storageConfig['rsync_options']);

        // Detect rsync version and add appropriate progress flag
        $progressFlag = $this->getRsyncProgressFlag();
        if ($progressFlag) {
            $rsyncOptions .= " {$progressFlag}";
        }

        $sshOptions = [];
        foreach ($sshConfig['options'] as $key => $value) {
            $sshOptions[] = "-o {$key}={$value}";
        }
        $sshOptionsString = implode(' ', $sshOptions);

        $remoteFullPath = "{$sshConfig['username']}@{$sshConfig['host']}:{$remotePath}";
        $localFullPath = base_path($localPath);

        return "rsync {$rsyncOptions} -e \"ssh {$sshOptionsString}\" {$remoteFullPath} {$localFullPath}";
    }

    protected function buildScpCommand(string $remotePath, string $localPath): string
    {
        $sshConfig = $this->config['ssh'];

        $sshOptions = [];
        foreach ($sshConfig['options'] as $key => $value) {
            $sshOptions[] = "-o {$key}={$value}";
        }
        $sshOptionsString = implode(' ', $sshOptions);

        $remoteFullPath = "{$sshConfig['username']}@{$sshConfig['host']}:{$remotePath}";

        return "scp {$sshOptionsString} {$remoteFullPath} {$localPath}";
    }

    protected function getDatabaseType(): string
    {
        return $this->config['database']['type'] ?? 'mysql';
    }

    protected function getStoragePaths(): array
    {
        $storageConfig = $this->config['storage'];

        if (empty($storageConfig['paths']) || ! is_array($storageConfig['paths'])) {
            throw new InvalidArgumentException('No storage paths configured. Please define storage paths in pull.storage.paths');
        }

        return array_map(function ($path, $index) {
            // Ensure each path has a name
            if (! isset($path['name'])) {
                $path['name'] = "path-{$index}";
            }

            return $path;
        }, $storageConfig['paths'], array_keys($storageConfig['paths']));
    }

    protected function getRsyncProgressFlag(): ?string
    {
        $result = Process::run('rsync --version');

        if ($result->successful() && str_contains($result->output(), ' version 3')) {
            return '--info=progress2';
        }

        return '--progress';
    }

    protected function validateDatabaseConfig(): void
    {
        $baseRequired = [
            'ssh.username',
            'ssh.host',
        ];

        foreach ($baseRequired as $key) {
            $value = data_get($this->config, $key);
            if (empty($value)) {
                throw new InvalidArgumentException("Missing required configuration: pull.{$key}");
            }
        }

        $databaseType = $this->getDatabaseType();

        if ($databaseType === 'sqlite') {
            $this->validateSqliteConfig();
        } else {
            $this->validateMysqlConfig();
        }
    }

    protected function validateSqliteConfig(): void
    {
        $required = [
            'database.sqlite.remote_path',
            'local_database.database',
        ];

        foreach ($required as $key) {
            $value = data_get($this->config, $key);
            if (empty($value)) {
                throw new InvalidArgumentException("Missing required configuration for SQLite: pull.{$key}");
            }
        }

        // Validate that local database is also SQLite
        $localConnection = $this->config['local_database']['connection'] ?? 'mysql';
        if ($localConnection !== 'sqlite') {
            throw new InvalidArgumentException('Local database must be SQLite when pulling from SQLite production database');
        }
    }

    protected function validateMysqlConfig(): void
    {
        $required = [
            'database.username',
            'database.password',
            'database.database',
            'local_database.username',
            'local_database.database',
        ];

        foreach ($required as $key) {
            $value = data_get($this->config, $key);
            if (empty($value)) {
                throw new InvalidArgumentException("Missing required configuration: pull.{$key}");
            }
        }
    }

    protected function validateStorageConfig(): void
    {
        $required = [
            'ssh.username',
            'ssh.host',
        ];

        foreach ($required as $key) {
            $value = data_get($this->config, $key);
            if (empty($value)) {
                throw new InvalidArgumentException("Missing required configuration: pull.{$key}");
            }
        }

        $storageConfig = $this->config['storage'];

        if (empty($storageConfig['paths']) || ! is_array($storageConfig['paths'])) {
            throw new InvalidArgumentException('No storage paths configured. Please define storage paths in pull.storage.paths');
        }

        foreach ($storageConfig['paths'] as $index => $path) {
            if (empty($path['remote_path'])) {
                throw new InvalidArgumentException("Missing remote_path for storage path at index {$index}");
            }
            if (empty($path['local_path'])) {
                throw new InvalidArgumentException("Missing local_path for storage path at index {$index}");
            }
        }
    }
}
