# Laravel Pull

A Laravel package that simplifies copying production data (database and storage files) to your local development environment. Perfect for developers who need to work with real production data locally.

## Features

- 🗄️ **Database Pulling**: Stream production database data directly to your local database using mysqldump over SSH
- 📁 **Storage Pulling**: Sync production storage files to local using rsync over SSH

## Installation

Add the following to your `composer.json` file:
```json
{
    "repositories": [
        {
            "type": "composer",
            "url": "https://package.firstpoint.ch"
        },
    ],
}
```

Then you can install the package via composer:

```bash
composer require firstpoint-ch/laravel-pull --dev
```

Publish the config file:

```bash
php artisan vendor:publish --provider="FirstpointCh\LaravelPull\LaravelPullServiceProvider"
```

## Configuration

### Storage Paths

Configure multiple storage paths to sync different directories from production:

```php
// config/pull.php
'storage' => [
    'paths' => [
        [
            'name' => 'public',
            'remote_path' => '~/sites/example.com/storage/app/public/',
            'local_path' => 'storage/app/public/',
        ],
    ],
],
```

### Environment Variables

Add the following environment variables to your `.env` file:

```env
# Production SSH Connection
PROD_SSH_USERNAME=your-ssh-username
PROD_SSH_HOST=your-production-server.com
PROD_SSH_PORT=22

# Production Database
PROD_DB_HOST=127.0.0.1
PROD_DB_PORT=3306
PROD_DB_DATABASE=production_database
PROD_DB_USERNAME=prod_db_user
PROD_DB_PASSWORD=prod_db_password
```

### SQLite Database Support

For SQLite databases, the package will copy the database file directly using SCP instead of using mysqldump. Configure it like this:

```env
# Production Database (SQLite)
PROD_DB_TYPE=sqlite
PROD_SQLITE_PATH=~/sites/yoursite.com/database/database.sqlite
```

Make sure your local database is also configured to use SQLite in your `config/database.php` and the pull configuration:

```php
// config/pull.php
'database' => [
    'type' => env('PROD_DB_TYPE', 'mysql'), // Set to 'sqlite' for SQLite

    // SQLite settings
    'sqlite' => [
        'remote_path' => env('PROD_SQLITE_PATH'),
    ],

    // ... other settings
],

'local_database' => [
    'connection' => env('DB_CONNECTION', 'mysql'), // Should be 'sqlite' for SQLite
    'database' => env('DB_DATABASE'),
    // ... other settings
],
```

**Note:** When using SQLite, both your production and local databases must be SQLite. The package will validate this configuration and throw an error if there's a mismatch.

### Database Types Supported

- **MySQL** (default): Uses `mysqldump` and `mysql` commands
- **SQLite**: Direct file copy using SCP

## Usage

### Artisan Commands

Pull everything (database + storage):
```bash
php artisan pull
```

Pull only database:
```bash
php artisan pull:database

# Skip confirmation prompt
php artisan pull:database --force
```

Pull only storage:
```bash
php artisan pull:storage

# Pull specific storage path
php artisan pull:storage public
```

## How It Works

### Database Pulling
1. Connects to your production server via SSH
2. Runs `mysqldump` with optimized settings on the production database
3. Streams the dump directly to your local MySQL database
4. Uses `REPLACE INTO` statements to handle duplicate keys gracefully

### Storage Pulling
1. Uses `rsync` over SSH to sync files
2. Automatically detects rsync version for optimal progress reporting
3. Preserves file permissions and timestamps
4. Removes local files that no longer exist in production (with `--delete`)

## Requirements

- PHP 8.3+
- Laravel 11.0+
- SSH access to production server
- `mysqldump` and `mysql` commands available
- `rsync` command available

## Security Considerations

- Store production credentials securely in environment variables
- Use SSH key authentication when possible
- Consider using read-only database users for pulling
- Be cautious when pulling to shared development environments
- The package includes confirmation prompts for destructive operations

## Testing

```bash
composer test
```
