# Cream



## Installation

```bash
composer require firstpoint-ch/cream
```

Ensuite publiez les assets

```bash
php artisan vendor:publish --provider="FirstpointCh\Cream\CreamServiceProvider"
```

**Frontend**

Si vous partez d'un nouveau projet, installez Tailwind:

```bash
npm install -D tailwindcss@latest postcss@latest postcss-import@latest autoprefixer@latest @tailwindcss/forms @tailwindcss/typography @tailwindcss/aspect-ratio

npx tailwindcss init
```

Voici un exemple de configuration:

```js
// tailwind.config.js
const defaultTheme = require('tailwindcss/defaultTheme');

module.exports = {
    mode: 'jit',

    purge: [
        './vendor/laravel/framework/src/Illuminate/Pagination/resources/views/*.blade.php',
        './vendor/firstpoint-ch/cream/resources/views/**/*.blade.php',
        './storage/framework/views/*.php',
        './resources/views/**/*.blade.php',
        './app/Http/Livewire/**/*.php', // Au cas où il y a du html dans le code
    ],

    theme: {
        extend: {
            fontFamily: {
                sans: ['Inter var', ...defaultTheme.fontFamily.sans],
            },
        },
    },

    variants: {},

    plugins: [
        require('@tailwindcss/forms'),
        require('@tailwindcss/typography'),
        require('@tailwindcss/aspect-ratio'),
    ],
};
```

```js
// webpack.mix.js

mix.js('resources/js/app.js', 'public/js')
    .postCss('resources/css/app.css', 'public/css', [
        require('postcss-import'),
        require('tailwindcss'),
        require('autoprefixer'),
    ]);
```

## Resources

### Préparer un model

Ajouter le trait

```php
use FirstpointCh\Cream\Traits\CreamResource;

class User extends Model
{
    use CreamResource;
}
```

### Policies

Par défaut, Cream intègre le système d'autorisations de Laravel pour certaines actions, il est donc nécessaire de créer des Policies pour chaque model avec lequel on souhaite interagir.

```bash
php artisan make:policy UserPolicy --model=App/Models/User
```

## Les Vues

### Formulaire de création

```bash
php artisan cream:view create Client/ClientCreate
```

### Formulaire de modification

```bash
php artisan cream:view edit Client/ClientEdit
```

### Fields

Chaque formulaire contient une méthode ```fields``` qui permet de définir les champs ainsi qu'une méthode ```save``` qui gère la logique d'enregistrement.

```php
use FirstpointCh\Cream\View\Components\Form\Text;

public function fields()
{
    return [
        Text::make('Name', 'name')->rules('nullable'),
    ];
}

public function save(array $data)
{
    // $data contient les données validées
    Client::create($data);

    return redirect()->route('app.clients.index');
}
```

### Types de champs

Il existe plusieurs types de champs mais tous partagent certaines méthodes:

- **make:** qui doit toujours être appelée en premier pour initialiser le champ.
- **rules:** qui permet de spécifier des règles de validation (en ```string``` ou en ```array```).
- **default**: qui permet de donner une valeur par défaut.
- **displayWhen**: qui reçoit un callback qui permet de savoir s'il faut afficher ou non le champ.
- **help**: qui permet d'afficher un message explicatif sous le label.

**BelongsTo**
Permet de rechercher un resources via une relation "BelongsTo". Elle s'utilise comme suit:

```php
// Affiche un <select> simple avec tous les clients (valeur = id, label = full_name)
BelongsTo::make('Client', 'client_id')->model(Client::class, 'full_name')

// Il est possible spécifier la colonne qui contient la valeur (valeur = uuid, label = full_name)
BelongsTo::make('Client', 'client_id')->model(Client::class, 'full_name', 'uuid')

// Affiche une dropdown avec la possibilité de rechercher par first_name, last_name ou company
BelongsTo::make('Client', 'client_id')
    ->model(Client::class, 'full_name')
    ->searchable(['first_name', 'last_name', 'company'])
```

**Date**
Affiche un champ ```<input type="date" />```

```php
Date::make('Date de publication', 'publish_date'),

// Valeur par défaut
Date::make('Date de publication', 'publish_date')->default(today()),
```

**Number**
Affiche un champ ```<input type="number" />```

```php
Number::make('Prix', 'price'),
```

**Password**
Affiche un champ ```<input type="password" />``` avec la possibilité de rendre le contenu visible

```php
Password::make('Mot de passe', 'password')
```

**Select**
Affiche un champ ```<select>```:

```php
Select::make('Type', 'type')
    ->options([
        'Web' => 'web',
        'Marketing' => 'marketing',
    ])
```

#### Text
Affiche un champ ```<input type="text" />```

```php
Text::make('Titre', 'title')
```

#### Textarea
Affiche un ```<textarea>```

```php
Textarea::make('Titre', 'title'); // 3 lignes par défaut

Textarea::make('Titre', 'title')->rows(10);
```

#### Toggle
Affiche un champs booléen

```php
Toggle::make('Actif', 'is_active');
```

#### Trix

Affiche un editeur Trix

```php
Trix::make('Contenu', 'content');
```

### Type de champ personnalisé

Il est également possible de créer ses propres types de champ. Pour cela il faut utiliser la commande suivante:

```bash
php artisan cream:field InvoiceItems
```

## Page d'index

La page d'index permet de lister, trier et rechercher des resources. Pour en créer une, il faut utiliser la commande suivante:

```bash
php artisan cream:view index Client/ClientIndex
```

### Colonnes

```php
use FirstpointCh\Cream\Column;

public function columns()
{
    return [
        Column::make('ID', 'id')->sortable(),

        // Custom output
        Column::make('Full name', 'full_name', function ($model) {
            return $model->fist_name . ' ' . $model->last_name;
        }),

        // HTML
        Column::make('Avatar', 'avatar', function ($model) {
            return '<img class="w-12 h-12 rounded-full" src="' . $model->picture_path . '" />';
        })->asHTML(),
    ]
}
```

### Recherche textuelle

```php
public $searchable = ['id', 'first_name', 'last_name', 'email'];
```

### Actions

```php
use FirstpointCh\Cream\View\Components\HeaderActions\ButtonLink;

public function heading()
{
    return [
        new ButtonLink('Créer un client', route('app.clients.create')),
    ];
}
```

### Inline actions

```php
public function actions($model)
{
    return [
        InlineAction::make('Export PDF', route('app.invoices.show', $model->id))
            ->icon('<svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path></svg>')
    ];
}
```

### Filtres

```php
use FirstpointCh\Cream\Filters\SelectFilter;

public function filters()
{
    return [
        SelectFilter::make('ID', 'id')->options([
            'Test 1' => 1,
            'Test 2' => 2,
        ]),
    ];
}
```

#### Personnaliser la query

```php
public function filters()
{
    return [
        NumberFilter::make('Age minimum', 'age')->handle(function ($query, $value) {
            $query->where('age', '>=', $value);
        }),
    ];
}
```

## Navigation

```php
// app/Providers/AppServiceProvider.php

use FirstpointCh\Cream\Cream;

public function boot(Cream $cream)
{
    $cream->navigation(function ($nav) {
        $nav->item('Dashboard', '/'); // Default icon
        $nav->item('Clients', '/clients')->icon('<svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z"></path></svg>');

        $nav->separator();

        $nav->item('Settings', '/settings');
    });
}
```

## Spotlight

Spotlight permet de déclencher des actions via une omnibar.

### Configuration

Publiez la configuration

```bash
php artisan vendor:publish --tag=livewire-ui-spotlight-config
```

### Doc

[https://github.com/livewire-ui/spotlight]()
