<?php

namespace FirstpointCh\Cream\Http\Livewire\Views;

use Livewire\Component;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;

abstract class IndexView extends Component
{
    use AuthorizesRequests;

    public $perPage = 15;

    public $sortBy = 'id';

    public $sortDirection = 'DESC';

    public $searchQuery;

    public $filters = [];

    public $displayHeader;

    public $attributes = [];

    public function mount($displayHeader = true, ...$attributes)
    {
        $this->authorize('viewAny', $this->model);

        $this->displayHeader = $displayHeader;
        $this->attributes = $attributes;
    }

    public function query()
    {
        $query = $this->model::query();

        $this->applySearchQuery($query);
        $this->applyFilters($query);
        $this->applySorting($query);

        return $query;
    }

    public function render()
    {
        return view('cream::livewire.views.index-view', [
            'columns' => $this->columns(),
            'models' => $this->query()->paginate($this->perPage),
        ])->layout('cream::layouts.app');
    }

    public function changeSortBy(string $property)
    {
        if ($this->sortBy === $property) {
            $this->sortDirection = $this->sortDirection === 'ASC' ? 'DESC' : 'ASC';
        } else {
            $this->sortBy = $property;
        }
    }

    public function filters()
    {
        return [];
    }

    private function applySearchQuery($query)
    {
        if (empty($this->searchQuery)) {
            return;
        }

        $query->where(function ($query) {
            foreach ($this->searchable as $field) {
                $query->orWhere($field, 'like', '%' . $this->searchQuery . '%');
            }
        });
    }

    private function applyFilters($query)
    {
        if (empty($this->filters())) {
            return;
        }

        foreach ($this->filters() as $filter) {
            if (!array_key_exists($filter->key, $this->filters)) {
                continue;
            }

            $value = $filter->getValue($this->filters[$filter->key] ?? null);

            if (is_null($value) && !$filter->allowNull) {
                return;
            }

            if (!empty($filter->handler)) {
                ($filter->handler)($query, $value);
            } else {
                $query->where($filter->key, $value);
            }
        }
    }

    private function applySorting($query)
    {
        if (!empty($this->sortBy)) {
            $query->orderBy($this->sortBy, $this->sortDirection);
        }
    }

    public function actions($model)
    {
        return [];
    }
}
